import Joi from 'joi';
import express from 'express';
import authenticator from 'authenticator';

import { sanitize } from '../../middlewares/sanitizer';
import { jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';

const validator = Joi.object().keys({
    otp: Joi.string().required().min(6).max(6)
});

const getQR = async (req, res, next) => {
  try {
    if (!req.user) {
      return res.status(401).json({ error: "Unauthorized" });
    }

    const formattedKey = authenticator.generateKey();
    const uri = authenticator.generateTotpUri(
      formattedKey,
      req.user.email,
      "TPIR",
      'SHA1',
      6,
      30
    );

    await req.user.updateOne({
      $set: { 'security.tOtpKey': formattedKey }
    });

    return res.status(200).json({ success: true, uri });
  } catch (error) {
    console.error('getQR error:', error);
    return next(error);
  }
};

const verifyQR = async (req, res, next) => {
  try {
    if (!req.user) {
      return res.status(401).json({ error: "Unauthorized" });
    }

    const { otp } = req.body;
    const isVerify = authenticator.verifyToken(req.user.security.tOtpKey, otp);

    if (!isVerify) {
      return next(new RequestError('Invalid OTP received', 400));
    }

    await req.user.updateOne({
      $set: {
        'security.status': true,
        'security.otpType': 'TOTP',
        'security.otp': null,
        'security.otpExpiry': null
      }
    });

    return res.status(200).json({
      success: true,
      message: 'Two-factor (TOTP) enabled successfully'
    });
  } catch (error) {
    console.error('verifyQR error:', error);
    return next(error);
  }
};


const apiRouter = express.Router();
apiRouter.route('/').get(sanitize(), jwtStrategy, getQR);
apiRouter.route('/').post(htmlsanitize(), jwtStrategy, validateBody(validator), verifyQR);
export default apiRouter;