import Joi from 'joi';
import express from 'express';
import bcrypt from 'bcryptjs';

import userActivity from '../../service/audit-log';
import { permission } from '../../middlewares/permission'; 
import { sanitize } from '../../middlewares/sanitizer';
import { jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';

let validator = Joi.object().keys({
    password: Joi.string().required(),
    currentpassword: Joi.string().required(),
});

/**
 * @api {post} /api/admin/change-password Change Password
 * @apiName change-password
 * @apiGroup Authentication
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {String} current password    Required current password of the User.
 * @apiBody {String} password            Required new password of the User.
 * 
 *
 * @apiSuccess (201) {String} success     true
 * @apiSuccess (201) {String} Message     Password Updated Successfully!
 * 
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *      "success":true,
 *      "Message":"Password Updated Successfully!"
 *      }
 * 
 * @apiError (404)  {String} status  false
 * @apiError (404)  {String} message Admin Not Found!
 * 
 * @apiErrorExample Error-Response:
 *     HTTP/1.1 404 Not Found
 *     {
 *      "success":false,
 *      "messages":["Admin Not found!"]
 *      }
 *
 * 
 * @apiError (500)  {String} status  false
 * @apiError (500)  {String} message Current Password Does not match!
 * 
 * @apiErrorExample Error-Response:
 *     HTTP/1.1 500 Not Found
 *     {
 *      "success":false,
 *      "messages":["Current Password Does not match!"]
 *      }
 */

let controller = async (req, res, next) => {
    try {
        const user = req.user;
        const { currentpassword, password } = req.body;
        const passwordMatch = bcrypt.compareSync(currentpassword, user.password) 
        
        const isSamePassword = bcrypt.compareSync(password, user.password)
        if(isSamePassword) throw new RequestError('New password must differ from the current password.')

        if(!passwordMatch) {
            throw new RequestError('Current Password Does not match!')
        }
        user.set({
            password: bcrypt.hashSync(password),
            loggedIn: false
        })
        await user.save();
        
        userActivity.create(req, 'Forgot Password');
        logger.info(`User ${user.email} Change Password`, { client: user.companyId, request: req, event: 'Change Password' });

        return res.status(201).send({ success: true, Message: "Password Updated Successfully!" })

    } catch (error) { 
        console.log(error);
        return next(new RequestError(error))
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(),  validateBody(validator),  jwtStrategy ,permission('Users','Write'), controller);
export default apiRouter;

