import Joi from 'joi';
import express from 'express';

import mailer from '../../mailer';
import { randomString } from '../../functions';
import { sanitize } from '../../middlewares/sanitizer';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';

const validator = Joi.object().keys({
    companyId: Joi.string().hex().length(24).required(),
	email: Joi.string().email().required(),
    type: Joi.any().valid('root', 'employee').required(),
});

/**
 * @api {post} /api/admin/check-company Check Company
 * @apiName check-company
 * @apiGroup Authentication
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {String} email                      Required email of the User.
 * @apiBody {String=root,employee} type=root    Required type of the User.
 * @apiBody {String} companyId                  Required companyId of the User.
 * 
 * 
 * @apiSuccess {Boolean} success   true
 * @apiSuccess {String}  message   successfully validate company
 * @apiSuccess {Boolean} isOtp     false
 * @apiSuccess {String}  otpType   Type of Otp
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *    {
 *       "success": true,
 *       "message": "successfully validate company",
 *       "isOtp": false,
 *       "otpType": null
 *      }
 * 
 * 
 * @apiError (400)  {String} status  false
 * @apiError (400)  {String} message \"companyId\" length must be 24 characters long
 * 
 * @apiErrorExample Error-Response:
 *     HTTP/1.1 400 Not Found
 *     {
 *      "success":false,
 *      "messages":["\"companyId\" length must be 24 characters long"]
 *      }
 * 
 *  @apiError (404)  {String} status false
 *  @apiError (404)  {String} message Email not found in TPIR database please register through this email and then use.
 * 
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 404 Not Found
 *     {
 *      "success":false,
 *      "messages":["Email not found in TPIR database please register through this email and then use"]
 *      }
 */

const controller = async (req, res, next) => {
    try {    
        const { email, type, companyId } = req.body;

        const company = await db.Company.findOne({_id:companyId, deletedAt: null})

        let user = null;
        if(type == 'root'){
            user = await db.Admin.findOne({ 
                email, 
                companyId, 
                deletedAt: null 
            })
        } else {
            user = await db.Employee.findOne({ 
                email, 
                companyId,
                deletedAt: null
            })
        }
        if(!user){
            throw new RequestError('Email not found in TPIR database please register through this email and then use', 404)
        }
        let message = 'successfully validate company',
            isOtp = false, otpType = null;
        
        if(user.security && user.security.status){
            isOtp = true
            if(user.security.otpType == 'MAIL'){
                const otp = randomString(6, '0123456789');
                const otpExpiry = new Date();
                await mailer.sendOTP(user.firstName ,user.email, otp, companyId);
                user.set({
                    'security.otp': otp,
                    'security.otpExpiry': otpExpiry
                })
                await user.save();
                otpType = 'MAIL';
                message = "OTP has been sent to your emailId";
            } else {
                otpType = 'TOPT';
                message = "Please check OTP in your Authenticator App";
            }
        }
        return res.status(200).json({
        success: true, message, isOtp, otpType, showLoginViaTa: {show: company?.isTacEnabled, companyName: company.name, alias: company.alias}
        });
        
    } catch (err) {
        console.log(err);
        next(err);
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), validateBody(validator), controller);
export default apiRouter;