import Joi from 'joi';
import express from 'express';

import crypto from '../../crypto';
import mailer from '../../mailer';
import { randomString } from '../../functions';
import userActivity from '../../service/audit-log';
import { validateBody } from '../../middlewares/validator';
import { sanitize } from '../../middlewares/sanitizer';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';

let validator = Joi.object().keys({
	email: Joi.string().email().required(),
	companyId: Joi.string().hex().length(24).required(),
	type: Joi.any().valid('root', 'employee'),
});

/**
 * @api {post} /api/admin/forgot-password Forgot Password
 * @apiName forgot-password
 * @apiGroup Authentication
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {String} email                      Required email of the User.
 * @apiBody {String=root,employee} type=root    Required type of the User.
 * @apiBody {String} companyId                  Required companyId of the User.
 * 
 * 
 * @apiSuccess {String} success   true
 * @apiSuccess {String} message   password reset link has been sent to you emailId
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *      "success":true,
 *      "message":"password reset link has been sent to you emailId"
 *      }
 * 
 * @apiError (404)  {String} status false
 * @apiError (404)  {String} message Email not found in TPIR database please register through this email and then use.
 * 
 * @apiErrorExample Error-Response:
 *     HTTP/1.1 404 Not Found
 *     {
 *      "success":false,
 *      "messages":["Email not found in TPIR database please register through this email and then use"]
 *      }
 */



let controller = async (req, res, next) => {
    try {
        const { email, type = 'root', companyId } = req.body;
        let user = null, token = randomString(20);
        if(type == 'root'){
            user = await db.Admin.findOne({ 
                email, 
                companyId,
                deletedAt: null
            })
        } else {
            user = await db.Employee.findOne({ 
                email, 
                companyId,
                deletedAt: null
            })
        }
        if(!user){
            throw new RequestError('invalid emailId recieved');
        }

        user.set({
            verfKey: token,
            verfKeyExpiry: new Date()
        })
        await user.save();

        let encryptEmail = await crypto.encrypt(email);
        encryptEmail = encodeURIComponent(encryptEmail);
        await mailer.sendResetLink(user.firstName, email, encryptEmail, token, companyId);        

        req['user'] = user;
        userActivity.create(req, 'Forgot Password');
        logger.info(`User ${email} Forgot Password`, { client: user.companyId, request: req, event: 'Forgot Password' });

        return res.status(200).send({ success: true, message: 'password reset link has been sent to you emailId' })
    }   
    catch (error) {
        console.log(error);
        next(new RequestError(error))
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), validateBody(validator), controller);
export default apiRouter;