import Joi from 'joi';
import express from 'express';
import JWT from 'jsonwebtoken';

import config from '../../config';
import userActivity from '../../service/audit-log';
import { sanitize } from '../../middlewares/sanitizer';
import { validateBody } from '../../middlewares/validator';
import { localStrategy } from '../../middlewares/strategy';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';

let JWTSign = function (iss, user, date, type, sudoPassword, companyFeatures) {
    return JWT.sign({
        iss: config.app.name + '-' + iss,
        sub: user._id,
        type: type,
        iat: date.getTime(),
        AF:companyFeatures,
        exp: new Date().setMinutes(date.getMinutes() + 600),
        sudoPassword: sudoPassword ? sudoPassword : false
    }, config.app.secret);
}

let validator = Joi.object().keys({
	email: Joi.string().email().required(),
    password: Joi.string().required(),
    companyId: Joi.string().hex().length(24).required(),
    type: Joi.any().valid('root', 'employee'),
    otp: Joi.string().optional().allow(null).empty(''),
});

/**
 * @api {post} /api/admin/login Login
 * @apiName LogIn
 * @apiGroup Authentication
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {String} email       Required email of the User.
 * @apiBody {String} password    Required password of the User.
 * 
 *
 * @apiSuccess {String} success   true
 * @apiSuccess {String} token     token
 * @apiSuccess {String} type      type
 * @apiSuccess {String} name      name
 * @apiSuccess {String} message   successfully login
 * 
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {"success":true,
 *      "token":"eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJUQUItVVNFUiIsInN1YiI6IjYzYWQ0NmNmNmI0MjlkYWE3YWE5YjY5OSIsInR5cGUiOiJyb290IiwiaWF0IjoxNjc0NTYwNTc5OTA3LCJleHAiOjE2OTI1NjA1Nzk5MDd9.4i8NVyOIT_P6lmWg3ZpCBzR9F7JvKkxDXqDYP804pq0",
 *      "type":"root",
 *      "name":"Sarvesh Patel",
 *      "message":["successfully login"]
 *      }
 * 
 * @apiError (404)  {String} status false
 * @apiError (404)  {String} message Email not found in TPIR database please register through this email and then use.
 * 
 * @apiErrorExample Error-Response:
 *     HTTP/1.1 404 Not Found
 *     {
 *      "success":false,
 *      "messages":["Email not found in TPIR database please register through this email and then use"]
 *      }
 */

let controller = async (req, res, next) => {
    const { type } = req.body;

    // if(!req.user.sudoPassword){
    //     userActivity.create(req, 'Login');
    //     logger.info(`User ${req.user.email} logged in`, { client: req.user.companyId, request: req, event: 'Logged In' });
    // }
    const company = await db.Company.findById(req.user.companyId);
    let companyFeatures = await db.CompanyFeature.find({companyId: company._id}).select({_id:0, featureCode:1})
    companyFeatures = companyFeatures.map(feature=>feature.featureCode)  

    let token = JWTSign('USER', req.user, new Date(), type, req.user.sudoPassword, companyFeatures);
    res.cookie('login_token', token, {
        httpOnly: true,
        secure: true,   
        sameSite: 'Strict',
        maxAge: 180 * 60 * 1000
    });
    return res.status(200).json({
        success: true,
        token: true,
        type: type,
        emai: req.user.email,
        name: req.user.firstName,
        isURL: company.logo?true:false,
        loginVia: 'TOOL',
        url: `${config.app.url}/api/static/logo/${company.logo}`,
        isTacEnabled: company.isTacEnabled ? company.isTacEnabled : false,
        message: ["successfully login"],
    });
}
const apiRouter = express.Router();
apiRouter.route('/').post( htmlsanitize(), validateBody(validator), localStrategy, controller);
export default apiRouter;

