import Joi from 'joi';
import express from 'express';
import bcrypt from 'bcryptjs';

import mailer from '../../mailer';
import { randomString } from '../../functions';
import userActivity from '../../service/audit-log';
import { sanitize } from '../../middlewares/sanitizer';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import registerCompanyTa from '../../service/register-company-ta';
import packageHelper from '../../service/package-helper';
import config from '../../config';

const validator = Joi.object().keys({
    firstName: Joi.string().required(),
    lastName: Joi.string().required(),
    companyName: Joi.string().required(),
	email: Joi.string().email().required(),
    industry: Joi.string().required(),
	country: Joi.string().required(),
    // password: Joi.string().min(8).max(16).required(),
    // confirmPassword: Joi.any().valid(Joi.ref('password')).required()
});

/**
 * @api {post} /api/admin/register Sign Up
 * @apiName register
 * @apiGroup Authentication
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {String} name               Required name of the User.
 * @apiBody {String} companyName        Required companyName of the User.
 * @apiBody {String} email              Required email of the User.
 * @apiBody {String} password           Required password of the User.
 * @apiBody {String} confirmPassword    Required password of the User.
 * 
 * 
 * @apiSuccess {String} success   true
 * @apiSuccess {String} message   User successfully registered and verification mail has been send to your provided emailId
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {"success":true,
 *      "message":"User successfully registered and verification mail has been send to your provided emailId"
 *      }
 * 
 * @apiError (404)  {String} status false
 * @apiError (404)  {String} message ["Email already exists."] 
 * 
 * @apiErrorExample Error-Response:
 *     HTTP/1.1 403 Not Found
 *     {
 *      "success":false,
 *      "messages":["Email already exists."]
 *      }
 */

const controller = async (req, res, next) => {
    try {    

        const { firstName, lastName, email, companyName, industry, country } = req.body;
        const verfKey = randomString(10);
        const user = await db.Admin.findOne({ 
            email,
            deletedAt: null
        })
        if(user){
            throw new RequestError('Email already exists.', 403)
        }

        const isCompanyExist = await db.Company.findOne({
            name: companyName
        })

        if(isCompanyExist) throw new RequestError('Company already exists.', 403)


        // await mailer.sendRegisterationLink(firstName, email, verfKey, company._id);

        
        const { companyDetails, licenseDetails, packageDetails }  = await registerCompanyTa({
            appName: "TPIR",
            companyName: companyName,
            firstName: firstName,
            lastName: lastName,
            email: email,
            industry,
            country
        })
        
        if(!companyDetails || !licenseDetails || !packageDetails) throw new RequestError("Error registering company!")

        const company = await db.Company.create({
            alias: companyDetails?.alias,
            name: companyName,
            tacLicenseKey: licenseDetails?.licenseKey,
            alias: companyDetails?.alias,
            tacCompanyId: companyDetails?.tacCompanyId,
            packageAccessKey: packageDetails?.packageAccessKey,
            isTacEnabled: true
        })

        const admin = await db.Admin.create({
            firstName, lastName, email, 
            companyId: company._id, verfKey, isTacEnabled: true,
            isActive: true,
            isVerify: true
        })
        

        const metaData = {
            tsatUrl: `${config.app.tsatUrl}`,
            isTsat: "false",
            mailSmtp: false,
            expiredAt: licenseDetails?.endDate,
            userLimit: licenseDetails?.emailUsersCount,
            userLimitMessenger: licenseDetails?.whatsappUsersCount,
            domainLimit: licenseDetails?.domainCount,
        }

        for (let key in metaData) {
            const update = await db.MetaData.findOneAndUpdate({
                companyId: company._id,
                name: key,
            },{
                value: metaData[key]
            },
            { upsert: true, new: true, setDefaultsOnInsert: true })
            console.log({update, key})
        }


        await db.Group.create({
            groupName: "all-users-tpir",
            type:"Custom",
            isDefaultGroup: true,
            emailAccess: false,
            messengerAccess: false,
            companyId: company._id,
            createdAt: new Date('9999-12-31')
        })

        const setPackage = await packageHelper.syncPackageDetails(company)

        req['user'] = admin;
        userActivity.create(req, 'Registerd');
        logger.info(`User ${email} register`, { client: company._id, request: req, event: 'Registerd' });

        return res.status(200).json({
            success: true,
            message: "User successfully registered and verification mail has been send to your provided emailId"
        });
        
    } catch (err) {
        console.log(err);
        next(err);
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), validateBody(validator), controller);
export default apiRouter;