import Joi from 'joi';
import express from 'express';
import bcrypt from 'bcryptjs';

import crypto from '../../crypto';
import { sanitize } from '../../middlewares/sanitizer';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import mailer from '../../mailer';

const validator = Joi.object().keys({
    token: Joi.string().required(),
	email: Joi.string().required(),
    password: Joi.string().min(8).max(16).required(),
    confirmPassword: Joi.any().valid(Joi.ref('password')).required(),
    isResettingPassword: Joi.boolean().optional()
});

/**
 * @api {post} /api/admin/reset-password Reset Password
 * @apiName reset-password
 * @apiGroup Authentication
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {String} token               Required token sent in mail.
 * @apiBody {String} email               Required email of the user.
 * @apiBody {String} password            Required new password of the User.
 * @apiBody {String} password            Required confirm password of the User.
 *
 * @apiSuccess (200) {String} success     true
 * @apiSuccess (200) {String} Message     Password Updated Successfully!
 * 
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *      "success":true,
 *      "Message":"Password Updated Successfully!"
 *      }
 * 
 * @apiError (500)  {String} success   false
 * @apiError (500)  {Array}  messages  Invalid emailId or token recieved
 *
 * @apiErrorExample Error-Response:
 *     HTTP/1.1 500 Not Found
 *  {
 *  "success": false,
 *  "messages": [
 *      {
 *          "status": 500,
 *          "errorList": [
 *              "Invalid emailId or token recieved"
 *          ]
 *      }
 *   ]
 *   }
 *
 * 
 * @apiError (500)  {String} status  false
 * @apiError (500)  {String} message password reset link has been expired
 * 
 * @apiErrorExample Error-Response:
 *     HTTP/1.1 500 Not Found
 *  {
 *  "success": false,
 *  "messages": [
 *      {
 *          "status": 500,
 *          "errorList": [
 *              "password reset link has been expired"
 *          ]
 *      }
 *   ]
 *   }
 */

let controller = async (req, res, next) => {
      try {     
        const { email, token, password, isResettingPassword = false } = req.body;
        let user = null, type = 'root';
        let encryptEmail = await crypto.decrypt(email);
        encryptEmail = decodeURIComponent(encryptEmail);
        user = await db.Admin.findOne({ email: encryptEmail, verfKey: token });
        if(!user){
            type = 'employee';
            user = await db.Employee.findOne({ email: encryptEmail, verfKey: token });
        }
        if(!user){
            throw new RequestError('Invalid emailId or token recieved');
        }
        if(user.verfKeyExpiry.getTime() < new Date().getTime() - (30 * 60 * 1000)){
            throw new RequestError('password reset link has been expired');
        }

        if(user.password){
            const isSamePassword = bcrypt.compareSync(password, user.password)
            if(isSamePassword) throw new RequestError('New password must differ from the current password.')
        }

        user.set({ 
            attempt: 0,
            verfKey: null,
            isVerify: true,
            isActive: true,
            verfKeyExpiry: null,
            password: bcrypt.hashSync(password),
            loggedIn: false
        })
        await user.save();
        if(!isResettingPassword) await mailer.sendSuccessRegister(user.name, encryptEmail, user.companyId)
        return res.status(200).json({ success: true, message: 'Password Updated Successfully!' });
    } 
    catch (error) {
        console.log(error);
        next(new RequestError(error))
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post( htmlsanitize(), validateBody(validator), controller);
export default apiRouter;