import Joi from 'joi';
import axios from 'axios'
import express from 'express';
import JWT from 'jsonwebtoken';

import config from '../../config';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';

let JWTSign = function (iss, user, date, type, sudoPassword, companyFeatures) {
    return JWT.sign({
        iss: config.app.name + '-' + iss,
        sub: user._id,
        type: type,
        iat: date.getTime(),
        AF:companyFeatures,
        exp: new Date().setMinutes(date.getMinutes() + 600),
        sudoPassword: sudoPassword ? sudoPassword : false
    }, config.app.secret);
}

let validator = Joi.object().keys({
    tacToken: Joi.string().required(),
});

const validateTacToken = function(token){
    return new Promise(async (resolve, reject) => {
        let axiosConfig = {
            method: 'POST',
            url: `${process.env.TAC_URL}/api/auth/validate`,
            headers: { 
                Authorization: `Bearer ${token}`
            },
            data: {
                applicationName: 'TPIR'
            },
            ...(config.proxy.enableForTA && { httpsAgent: config.proxy.proxyAgent, httpAgent: config.proxy.proxyAgent })
        };
        axios.request(axiosConfig)
        .then(res => {
          resolve({ userId: res.data.response.userId, companyId: res.data.response.companyId, type: res.data.response.type });
        })
        .catch(err => {
            reject(err)
        })
    })
}

/**
 * @api {post} /api/admin/login Login
 * @apiName LogIn
 * @apiGroup Authentication
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {String} email       Required email of the User.
 * @apiBody {String} password    Required password of the User.
 * 
 *
 * @apiSuccess {String} success   true
 * @apiSuccess {String} token     token
 * @apiSuccess {String} type      type
 * @apiSuccess {String} name      name
 * @apiSuccess {String} message   successfully login
 * 
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {"success":true,
 *      "token":"eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJUQUItVVNFUiIsInN1YiI6IjYzYWQ0NmNmNmI0MjlkYWE3YWE5YjY5OSIsInR5cGUiOiJyb290IiwiaWF0IjoxNjc0NTYwNTc5OTA3LCJleHAiOjE2OTI1NjA1Nzk5MDd9.4i8NVyOIT_P6lmWg3ZpCBzR9F7JvKkxDXqDYP804pq0",
 *      "type":"root",
 *      "name":"Sarvesh Patel",
 *      "message":["successfully login"]
 *      }
 * 
 * @apiError (404)  {String} status false
 * @apiError (404)  {String} message Email not found in TPIR database please register through this email and then use.
 * 
 * @apiErrorExample Error-Response:
 *     HTTP/1.1 404 Not Found
 *     {
 *      "success":false,
 *      "messages":["Email not found in TPIR database please register through this email and then use"]
 *      }
 */

let controller = async (req, res, next) => {
    try {
        const { tacToken } = req.body;
        let { userId, companyId, type } = await validateTacToken(tacToken)
    
        
        let user;
        if(type=='root') user = await db.Admin.findOne({_id: userId, companyId})
        else user = await db.Employee.findOne({_id: userId, companyId})
        
        if(!user) throw new RequestError('User not found!',404)
        
        let company = await db.Company.findById(companyId)
        if(!company) throw new RequestError('Company not found!',404)

        let companyFeatures = await db.CompanyFeature.find({companyId: company._id}).select({_id:0, featureCode:1})
        companyFeatures = companyFeatures.map(feature=>feature.featureCode)  
        
        let token = JWTSign('USER', user, new Date(), type, false, companyFeatures);
        

        if(type=='root') await db.Admin.updateOne({_id: userId},{loggedIn: true})
        else await db.Employee.updateOne({_id: userId},{loggedIn: true})
        
        res.cookie('login_token', token, {
            httpOnly: true,
            secure: true,   
            sameSite: 'Strict',
            maxAge: 180 * 60 * 1000
        });

        return res.status(200).json({
            success: true,
            token: true,
            email: user.email,
            isTacEnabled: company?.isTacEnabled,
            tacUrl: config.adminCentral.url,
            loginVia: 'TAC',
            type: type,
            name: user.firstName,
            isURL: company.logo?true:false,
            url: `${config.app.url}/api/static/logo/${company.logo}`,
            message: ["successfully login"]
        });
    } catch (error) {
        next(error)
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post( htmlsanitize(), controller);
export default apiRouter;

