import express from 'express';
import config from '../../config';
import getGraphData from '../../service/dashboardHelper';

const getGraphHeaders = {
    RETPIR1: ['Reporter', 'Mails Reported'],
    RETPIR2: ['Sender', 'Subject', 'Reporters', 'Date', 'Status'],
    RETPIR3: ['Url', 'Scan Score', 'Sender'],
    RETPIR4: ['Attachment', 'Scan Score', 'Sender'],
    RETPIR5: ['Sender', 'Total Emails', 'Receiver'],
    RETPIR6: [],
};

const getGraphUnit = {
    RETPIR1: 'ArrayOfArray',
    RETPIR2: 'ArrayOfArray',
    RETPIR3: 'ArrayOfArray',
    RETPIR4: 'ArrayOfArray',
    RETPIR5: 'ArrayOfArray',
    RETPIR6: 'ArrayOfArray',
};

const optimizeResult = {
    RETPIR1: (reporters) => reporters.map((reporter) => [[reporter.name || '', reporter.email || ''], reporter.count || 0]),

    RETPIR2: (emails) =>
        emails.map((email) => {
            const emailStatus = Object.entries({
                isRecoveredEmail: email.isRecoveredEmail,
                isTrashedEmail: email.isTrashedEmail,
                isDeletedEmail: email.isDeletedEmail,
            }).find(([key, value]) => value === 'true');

            return [
                [email.fromName || '',email.from || ''],
                email.subject || '',
                email.totalReporters || 0,
                email.createdAt || 'Unknown Date',
                emailStatus ? emailStatus[0] : 'No Action',
            ];
        }),

    RETPIR3: (urls) =>
        urls.map((url) => [
            url.url || '',
            url.score || 0,
            [url.reportedMailId?.fromName || '', url.reportedMailId?.from || ''],
        ]),

    RETPIR4: (attachments) =>
        attachments.map((attachment) => [
            attachment.filename || '',
            attachment.score || 0,
            [attachment.reportedMailId?.fromName || '', attachment.reportedMailId?.from || '']
        ]),

    RETPIR5: (senders) =>
        senders.map((sender) => [
            [sender.name || '', sender.email || ''],
            sender.reportedMailCount || 0,
            sender.whoElseCount || 0,
        ]),

    RETPIR6: data => data
};

const controller = async (req, res, next) => {
    try {
        const { graphCode } = req.params;
        const { licenseKey, beginDate, endDate } = req.query;

        const authHeader = req?.headers?.authorization;
        const apiKey = authHeader?.split(' ')[1] ?? null;
        if (apiKey !== config.adminCentral.api_key) {
            throw new RequestError('Unauthorized User', 401);
        }

        const company = await db.Company.findOne({ tacLicenseKey: licenseKey });
        if (!company) {
            throw new RequestError('Company not found!', 404);
        }

        let condition = {
            companyId: company._id
        }
        if(beginDate.length) condition[startDate] = beginDate
        if(endDate.length) condition[endDate] = endDate

        const graphData = await getGraphData(graphCode, condition);

        if (!optimizeResult[graphCode] || !getGraphHeaders[graphCode]) {
            throw new RequestError('Invalid graph code!', 400);
        }

        const result = {
            fields: optimizeResult[graphCode](graphData?.data || []),
            unit: getGraphUnit[graphCode],
            graphCode,
            headers: getGraphHeaders[graphCode],
            label: graphData?.label || '',
            type: graphData?.type || '',
        };

        res.status(200).json({
            success: true,
            response: result,
        });
    } catch (error) {
        const statusCode = error.status || 500;
        const isClientError = statusCode >= 400 && statusCode < 500;

        console.error('API Error:', {
            message: error.message,
            stack: error.stack,
        });

        res.status(statusCode).json({
            success: false,
            message: isClientError
                ? error.message || 'The request could not be processed.'
                : 'An internal server error occurred.',
            error: isClientError
                ? {
                    code: error.code || 'UNEXPECTED_ERROR',
                    description: error.description || null,
                }
                : undefined,
        });
    }
};

const apiRouter = express.Router();
apiRouter.route('/:graphCode').get(controller);
export default apiRouter;
