import express from 'express';
import axios from 'axios';
import fs from 'fs';
import path from 'path';
import { jwtStrategy } from '../../middlewares/strategy';
import config from '../../config';

const logoDir = path.join(path.dirname(require.main.filename), 'public', 'logo');

async function taConnected(req, res, next) {
  try {
    const company = await db.Company
      .findById(req.user.companyId)
      .select('isTacEnabled')
      .lean();

    if (company?.isTacEnabled) return next();
    throw new RequestError('Threatcop Admin is not connected', 403);
  } catch (error) {
    next(error);
  }
}

let controller = async (req, res, next) => {
  try {
    const companyId = req.user.companyId;
    const company = await db.Company.findById(companyId);
    if (!company) throw new RequestError('Company not found', 404);

    const { tacLicenseKey } = company;
    if (!tacLicenseKey) throw new RequestError('License key not found', 404);

    const response = await axios.get(
      `${config.adminCentral.url}/api/sync-services/common/fetch-organisation-details`,
        { 
          headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${tacLicenseKey}`,
        },
        ...(config.proxy.enableForTA && { httpsAgent: config.proxy.proxyAgent, httpAgent: config.proxy.proxyAgent })
      }
    );
    if (!response.data?.response) throw new RequestError('Invalid response from external API', 502);
    console.log('fetchOrganisationDetails response:', response.data.response);
    const { companyDetails, rootUserDetails } = response.data.response;

    await db.Admin.updateOne(
      { companyId },
      {
        firstName: rootUserDetails.firstName,
        lastName:  rootUserDetails.lastName,
        phone:     rootUserDetails.phoneNumber,
      }
    );

    const remoteLogo = companyDetails.logo;
    console.log('remoteLogo:', remoteLogo);
    if (!remoteLogo) {
      return res.status(200).json({
        success: true,
        message: 'Org details synced; no logo provided.',
      });
    }
    const cleanPath  = remoteLogo.split('?')[0];
    const remoteBase = path.basename(cleanPath);
    const filename   = `${companyId}_${remoteBase}`;
    console.log('filename:', filename);
    if (company.logo === filename) {
      console.log('Logo already exists, skipping download.');
      return res.status(200).json({
        success: true,
        message: 'Org details synced; logo not changed.',
        logoPath: `${config.app.url}/api/static/logo/${filename}`,
      });
    }
    console.log('Downloading logo...');
    const imageUrl = `${config.adminCentral.url}/${remoteLogo}`;
    const axiosConfig = {
      responseType: 'stream'
    };

    if (config.proxy.enableForTA) {
      axiosConfig.httpsAgent = config.proxy.proxyAgent;
      axiosConfig.httpAgent = config.proxy.proxyAgent;
    }

    const imageResp = await axios.get(imageUrl, axiosConfig);

    if (imageResp.status !== 200) throw new RequestError('Failed to fetch logo', 502);
    console.log('Logo downloaded, saving to local storage...');
    const localPath = path.join(logoDir, filename);
    await new Promise((resolve, reject) => {
      const writer = fs.createWriteStream(localPath);
      imageResp.data.pipe(writer);
      writer.on('finish', resolve);
      writer.on('error', reject);
    });
    console.log('Logo saved to local storage.');
    company.logo = filename;
    await company.save();

    return res.status(200).json({
      success: true,
      message: 'Org details synced; logo updated.',
      logoPath: `${config.app.url}/api/static/logo/${filename}`,
    });
  } catch (error) {
    console.error('fetchOrganisationDetails error:', error);
    const statusCode = error.status || 500;
    const isClientError = statusCode >= 400 && statusCode < 500;
    return res.status(statusCode).json({
      success: false,
      message: isClientError ? error.message : 'Internal server error.',
      ...(isClientError && { error: { code: error.code || 'UNEXPECTED_ERROR' } }),
    });
  }
};

const apiRouter = express.Router();
apiRouter.route('/').get(jwtStrategy, taConnected, controller);
export default apiRouter;