import Joi from 'joi';
import express from 'express';

import { permission } from '../../middlewares/permission'; 
import { checkCompanyValidityStrategy, jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import restrictFeature from '../../middlewares/featureBlocker';

const validator = Joi.object().keys({
    page: Joi.number().min(0).required(),
    limit: Joi.number().min(1).required(),
    search: Joi.string().optional().allow(null).empty(''),
    advanceSearchId: Joi.string().hex().length(24).required()
});

/**
 * @api {get} /api/v1/advance-search/emails?page=1&limit=10&advanceSearchId=63b281595e8160f2ed081806 emails
 * @apiName emails
 * @apiGroup Advance Search
 * @apiDescription Fetch Record For Particular Search
 * 
 * @apiVersion 1.0.0
 * 
 * @apiQuery {String}          advanceSearchId      Particular Advance Search id.
 * @apiQuery {Number}          page                 Takes page number to fetch
 * @apiQuery {Number}          limit                Takes number of records to show in a page
 * 
 * @apiSuccess {Boolean} Success   true
 * @apiSuccess {String}  results   list of found emails for search.
 * @apiSuccess {Boolean} count     number of results returned
 *  
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *      {
 *        "Success":true,
 *        "results":[{
 *                       "_id": "63ca962c6e4d31196150f6c4",
 *                       "from": "vikas@kratikal.co",
 *                       "fromName": "vikas kumar",
 *                       "to": [
 *                           {
 *                               "emailAddress": {
 *                                   "name": "Himanshu Bisht",
 *                                   "address": "Himanshu@kdmarc.com"
 *                               }
 *                           },
 *                           {
 *                               "emailAddress": {
 *                                   "name": "vikas",
 *                                   "address": "vikas@kdmarc.com"
 *                               }
 *                           }
 *                               ],
 *                       "cc": [
 *                           {
 *                               "emailAddress": {
 *                                   "name": "Gaurav Pandey",
 *                                   "address": "gaurav@kdmarc.com"
 *                               }
 *                           },
 *                           {
 *                               "emailAddress": {
 *                                   "name": "Arpit Rao",
 *                                   "address": "arpit@kdmarc.com"
 *                               }
 *                           },
 *                           {
 *                               "emailAddress": {
 *                                   "name": "Pavan Kushwaha",
 *                                   "address": "pavan@kdmarc.com"
 *                               }
 *                           }
 *                               ],
 *                           "mailRecievedTime": "2023-01-19T05:21:53.000Z",
 *                           "body": "<html><head>\r\n<meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\"></head><body><div dir=\"ltr\"><div class=\"gmail_default\" style=\"font-family:verdana,sans-serif\">Hello Team,</div><div class=\"gmail_default\" style=\"font-family:verdana,sans-serif\"><br></div><div class=\"gmail_default\" style=\"font-family:verdana,sans-serif\">Hope everyone is doing well!!</div><div class=\"gmail_default\" style=\"font-family:verdana,sans-serif\"><br></div><div class=\"gmail_default\" style=\"font-family:verdana,sans-serif\">As you all are aware, we are in the process of getting complied with GDPR standards.</div><div class=\"gmail_default\" style=\"font-family:verdana,sans-serif\">GDPR dedicatedly talks about the Data security of the Data subjects.</div><div class=\"gmail_default\" style=\"font-family:verdana,sans-serif\"><br></div><div class=\"gmail_default\" style=\"font-family:verdana,sans-serif\"><br></div><div class=\"gmail_default\" style=\"font-family:verdana,sans-serif\"><div class=\"gmail_default\"><a href=\"https://www.google.com/\" target=\"_blank\">https://www.google.com/</a></div><div class=\"gmail_default\"><br></div><div class=\"gmail_default\"><a href=\"https://www.color-name.com/\" target=\"_blank\">https://www.color-name.com/</a><br></div><div class=\"gmail_default\"><br></div><div class=\"gmail_default\"><a href=\"https://icolorpalette.com/\" target=\"_blank\">https://icolorpalette.com/</a></div><div class=\"gmail_default\"><br></div><div class=\"gmail_default\"><a href=\"https://kratikal.com/\" target=\"_blank\">https://kratikal.com/</a></div></div></div></body></html>",
 *                           "subject": "New Specifications",
 *                           "messageId": "AAMkADg4N2I3ODU1LTg5MzgtNGJlOS05MDdmLTdkODQzY2UxODJlMABGAAAAAAAEeu5jAGlBSIzFYD0Uf00rBwDYeUh9annVS4gyZQy0RznfAAAAAAEMAADYeUh9annVS4gyZQy0RznfAABeS4CmAAA=",
 *                           "isTrashed": false,
 *                           "userId": "63b677a644ed3b8a723865b0",
 *                           "domainId": "63b676c744ed3b8a72386597",
 *                           "companyId": "63ad46cf6b429daa7aa9b697",
 *                           "advanceSearchId": "63ca962b6e4d31196150f6c0",
 *                           "deletedAt": null,
 *                           "__v": 0,
 *                           "createdAt": "2023-01-20T13:25:00.761Z",
 *                           "updatedAt": "2023-01-20T13:25:00.761Z"
 *                   }],
 *              "count": 1
 *       }
 * 
 *      @apiError (400)  {String} success-1    false
 *      @apiError (400)  {Array}  messages-1  "advanceSearchId length must be 24 characters long"
 *      @apiError (400)  {String} success-2    false
 *      @apiError (400)  {Array}  messages-2  "advanceSearchId must only contain hexadecimal characters"
 * 
 *      @apiErrorExample Error-Response:
 *         HTTP/1.1 400 Not Found
 *         {
 *          "success":false,
 *          "messages":["\"advanceSearchId\" length must be 24 characters long"]
 *          }
*      @apiErrorExample Error-Response:
 *         HTTP/1.1 400 Not Found
 *         {
 *          "success":false,
 *          "messages":["\"advanceSearchId\" must only contain hexadecimal characters"]
 *          }
 *
 */

const controller = async (req, res, next) => {
    try {
        const { advanceSearchId, page = 1, limit = 10, search = '' } = req.query;
        let condition = { 
            advanceSearchId,
            deletedAt: null
        };
        if(search && search.length){
            const regex = new RegExp(search, "i");
            condition['$or'] = [
                { to: regex }, 
                { from: regex }, 
                { subject: regex }, 
            ]
        }
        let count = await db.AdvanceSearchEmail.countDocuments(condition);
        let results = await db.AdvanceSearchEmail.find(condition)
        .skip((page-1)*limit)
        .limit(limit);
        return res.status(200).send({ Success: true, results, count })
    } catch (error) {
        console.log('error', error);
        next(error)
    }
}

const apiRouter = express.Router();
apiRouter.route('/').get(jwtStrategy, restrictFeature("TPIR-ADVS"),  permission('AdvancedSearch','Read'), validateBody(validator), controller);
export default apiRouter;