import Joi from 'joi';
import express from 'express';

import { permission } from '../../middlewares/permission';
import { jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import restrictFeature from '../../middlewares/featureBlocker';

const validator = Joi.object().keys({
    page: Joi.number().min(0).required(),
    limit: Joi.number().min(1).required(),
    domainId: Joi.string().hex().length(24).optional(),
    search: Joi.string().optional().allow(null).empty('')
});

/**
 * @api {get} /api/v1/advance-search/list?page=1&limit=10&domainId=63a2b5df5f536ee8fb0b2093 list
 * @apiName list
 * @apiGroup Advance Search
 * @apiDescription Fetch all advance searches
 * 
 * @apiVersion 1.0.0
 * 
 * @apiQuery {String}          domainId             Particular Domain id.
 * @apiQuery {Number}          [page]                 Takes page number to fetch
 * @apiQuery {Number}          [limit]                Takes number of records to show in a page
 * @apiQuery {String}          [search]             Takes a string to search a particular record
 * 
 * @apiSuccess {Boolean} Success   true
 * @apiSuccess {String}  results   list of all advance search(s).
 * @apiSuccess {Boolean} count     number of results returned
 *  
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *      {  "Success":true,
 *         "results":[
 *                  "_id": "63cfa68cfa94cabd3e6a596f",
 *                  "from": "adads@gmail.com",
 *                  "to": "",
 *                  "subject": "a",
 *                  "fromDate": "2023-01-23T00:00:00.000Z",
 *                  "toDate": "2023-01-24T23:59:59.000Z",
 *                  "filter": "&$filter=from/emailAddress/address eq 'adads@gmail.com' and subject eq 'a' and receivedDateTime ge 2023-01-23T00:00:00Z and receivedDateTime le 2023-01-24T23:59:59Z",
 *                  "isError": false,
 *                  "isErrorMessage": "",
 *                  "isProcessing": false,
 *                  "syncStartAt": "2023-01-24T09:36:12.537Z",
 *                  "domainId": "63b676c744ed3b8a72386597",
 *                  "companyId": "63ad46cf6b429daa7aa9b697",
 *                  "deletedAt": null,
 *                  "createdAt": "2023-01-24T09:36:12.539Z",
 *                  "updatedAt": "2023-01-24T09:36:30.789Z",
 *                  "__v": 0,
 *                  "syncEndAt": "2023-01-24T09:36:30.788Z",
 *                  "emailCount": 0
 *              ],
 *          "count":10
 *      }
 * 
 *      @apiError (400)  {String} success   false
 *      @apiError (400)  {Array}  messages "domainId length must be 24 characters long"
 * 
 *      @apiErrorExample Error-Response:
 *         HTTP/1.1 400 Not Found
 *         {
 *           "success": false,
 *           "messages": [
 *               "\"domainId\" length must be 24 characters long"
 *           ]
 *         }
 *
 */

const controller = async (req, res, next) => {
    try {
        const { domainId, page = 1, limit = 10, search = '' } = req.query;
        const companyId = req.user.companyId;
        let condition = { 
            companyId,
            deletedAt: null
        };
        if(domainId) condition['domainId'] = domainId
        if(search && search.length){
            const regex = new RegExp(search, "i");
            condition['$or'] = [
                { to: regex }, 
                { from: regex }, 
                { subject: regex }, 
            ]
        }
        let count = await db.AdvanceSearch.countDocuments(condition);
        let results = await db.AdvanceSearch.find(condition)
        .sort({'createdAt': -1})
        .skip((page-1)*limit)
        .limit(limit)
        for(let i=0; i<results.length; i++){
            let emailCount = await db.AdvanceSearchEmail.countDocuments({
                advanceSearchId: results[i]._doc._id
            });
            results[i] = { ...results[i]._doc, emailCount };
        }
        return res.status(200).send({ Success: true, results, count })
    } catch (error) {
        console.log('error', error);
        next(error)
    }
}

const apiRouter = express.Router();
apiRouter.route('/').get(jwtStrategy, restrictFeature("TPIR-ADVS"), permission('AdvancedSearch','Read'), validateBody(validator), controller);
export default apiRouter;