import Joi from 'joi';
import express from 'express';
import Credentials from '../../service/credentials';
import moveEmailOutlook from '../../service/move-email-outlook';
import { checkLicenseStrategy, jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { permission } from '../../middlewares/permission';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';

// Validator for the request body
const validator = Joi.object().keys({
    emails: Joi.array().items(
        Joi.object({
            email: Joi.string().email().required(),
            messageId: Joi.string().required()
        })
    ).required(),
    domainId: Joi.string().required()
});

/**
 * @api {post} /api/v1/advance-search/move-to-deleted Move Outlook Mails to DeletedItems
 * @apiName move-to-deleted
 * @apiGroup AdvanceSearch
 * @apiDescription Move Outlook mails to DeletedItems folder (Trash)
 *
 * @apiBody {Array} emails Array of objects with user email and messageId
 *
 * @apiSuccess {Boolean} success true
 * @apiSuccess {String} message Success message
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *         "success": true,
 *         "message": "Mails moved to DeletedItems successfully!"
 *     }
 */

const controller = async (req, res, next) => {
    try {
        const { emails, domainId } = req.body;
        console.log('Move to deleted emails:', emails, 'for domain:', domainId);
        const companyId = req.user.companyId;
        console.log("Company ID:", companyId);
        if (!emails || !emails.length) {
            return res.status(400).json({ success: false, message: 'No emails provided.' });
        }

        // Group by user email for batching tokens
        const emailMap = {};
        for (const item of emails) {
            if (!emailMap[item.email]) emailMap[item.email] = [];
            emailMap[item.email].push(item.messageId);
        }

        let allBatches = [];
        // Get token for domain first to fail early if credentials are invalid
       const { token } = await Credentials.getCredentials(companyId, domainId);
       console.log(token);

        for (const email in emailMap) {
            // Prepare batches for this user
            for (const messageId of emailMap[email]) {
                allBatches.push({
                    id: `${email}_${messageId}`,
                    email,
                    token: token.access_token,
                    messageId
                });
            }
        }

        // Batch in groups of 20 for Outlook API
        let responses = [];
        for (let i = 0; i < allBatches.length; i += 20) {
            const batch = allBatches.slice(i, i + 20);
            const result = await moveEmailOutlook.processMoveEmailTrash(batch);
            responses.push(result);
        }

        await db.AdvanceSearchEmail.updateMany(
        {
            $and: [
            { messageId: { $in: emails.map(e => e.messageId) } },
            { isTrashed: false }
            ]
        },
        { $set: { isTrashed: true } }
        );


        return res.status(200).json({ success: true, message: 'Mails moved to DeletedItems successfully!' });
    } catch (error) {
        console.log('Move to deleted error:', error);
        next(error);
    }
};

const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), jwtStrategy, checkLicenseStrategy, permission('Emails','Delete'), validateBody(validator), controller);
export default apiRouter;
