import Joi from 'joi';
import express from 'express';

import { permission } from '../../middlewares/permission';
import { jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import restrictFeature from '../../middlewares/featureBlocker';

// Validator to ensure we receive an array of AdvanceSearch _id(s)
const validator = Joi.object().keys({
	ids: Joi.array().items(Joi.string().hex().length(24)).min(1).required()
});

/**
 * @api {post} /api/v1/advance-search/remove-entry Remove Advance Search Entry(ies)
 * @apiName remove-entry
 * @apiGroup Advance Search
 * @apiDescription Soft delete Advance Search record(s) and all related email(s)
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {Array}  ids            Array of AdvanceSearch document _id(s) to be soft deleted.
 * 
 * @apiSuccess {Boolean} Success                       true
 * @apiSuccess {Number}  deletedAdvanceSearchCount     Number of AdvanceSearch records soft deleted
 * @apiSuccess {Number}  deletedEmailsCount            Number of AdvanceSearchEmail records soft deleted
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *        "Success": true,
 *        "deletedAdvanceSearchCount": 1,
 *        "deletedEmailsCount": 25
 *     }
 * 
 * @apiError (400)  {Boolean} success      false
 * @apiError (400)  {Array}   messages     Validation error messages (e.g. "ids[0] length must be 24 characters long")
 * 
 * @apiErrorExample Error-Response:
 *     HTTP/1.1 400 Bad Request
 *     {
 *        "success": false,
 *        "messages": ["\"ids[0]\" length must be 24 characters long"]
 *     }
 */

const controller = async (req, res, next) => {
	try {
		const { ids } = req.body;
		const companyId = req.user.companyId;

		// Ensure ids present after validation middleware (defensive)
		if(!ids || !ids.length) throw new RequestError('No Advance Search entries provided for deletion.', 404); 

		const now = new Date();
		// Soft delete AdvanceSearch documents scoped to company
		const advanceResult = await db.AdvanceSearch.updateMany({
			_id: { $in: ids },
			companyId,
			deletedAt: null
		}, { $set: { deletedAt: now } });

		// Soft delete related AdvanceSearchEmail documents
		const emailResult = await db.AdvanceSearchEmail.updateMany({
			advanceSearchId: { $in: ids },
			companyId,
			deletedAt: null
		}, { $set: { deletedAt: now } });

		const deletedAdvanceSearchCount = advanceResult.modifiedCount ?? advanceResult.nModified ?? 0;
		const deletedEmailsCount = emailResult.modifiedCount ?? emailResult.nModified ?? 0;

		return res.status(200).send({
			success: true,
            message: 'Advance Search entries and related emails deleted successfully.',
			deletedAdvanceSearchCount,
			deletedEmailsCount
		});
	} catch (error) {
		console.log('error', error);
		next(error);
	}
};

const apiRouter = express.Router();
apiRouter.route('/').post(jwtStrategy, restrictFeature('TPIR-ADVS'), permission('AdvancedSearch','Read'), validateBody(validator), controller);
export default apiRouter;
