import Joi from 'joi';
import express from 'express';

import { queue } from '../../kue';
import userActivity from '../../service/audit-log';
import { checkCompanyValidityStrategy, jwtStrategy } from '../../middlewares/strategy';
import { permission } from '../../middlewares/permission';
import { validateBody } from '../../middlewares/validator';
import { sanitize } from '../../middlewares/sanitizer';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import restrictFeature from '../../middlewares/featureBlocker';
import { removeStopwords } from 'stopword';

const validator = Joi.object().keys({
    from: Joi.string().email().required(),
    to: Joi.string().email().optional().allow(null).empty(''),
    subject: Joi.string().optional().allow(null).empty(''),
    fromDate: Joi.date().iso().required(),
    toDate: Joi.date().iso().required(),
    domainId: Joi.string().hex().length(24).required()
});

/**
 * @api {post} /api/v1/advance-search/search search
 * @apiName search
 * @apiGroup Advance Search
 * @apiDescription Make a search for mail by someone between two dates with a particular subject
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {String}  from        Email of sender
 * @apiBody {String}  [to]        Email of reciever
 * @apiBody {String}  subject     Subject of mail
 * @apiBody {String}  fromDate    Start date of search
 * @apiBody {String}  toDate      End date of search
 * @apiBody {String}  domainId    Domain id in which search is to be done
 *
 * 
 * @apiSuccess {Boolean} Success   true
 * @apiSuccess {String}  message   Searching start successfully
 *  
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *       {
 *           "success": true,
 *           "message": "Searching start successfully"
 *       }
 * 
 *  @apiError (400)  {Boolean} success-1    false
 *  @apiError (400)  {Array}  messages-1   invalid domain recieved
 *  @apiError (400)  {Boolean} success-2    false
 *  @apiError (400)  {Array}  messages-2   "subject" is not allowed to be empty
 * 
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 400 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *               "invalid domain recieved"
 *           ]
 *       }
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 400 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *              "\"subject\" is not allowed to be empty"
 *           ]
 *       }
 */
const isNonEmpty = (v) => typeof v === 'string' && v.trim().length > 0;
const escapeOData = (s = '') => s.replace(/'/g, "''");

// Build $filter for Outlook (Graph) only. No $search here.
function buildOutlookFilter({ fromDate, toDate, from, to, subject }) {
  const parts = [];

  const f = new Date(fromDate);
  const t = new Date(toDate);
  if (Number.isNaN(f.getTime()) || Number.isNaN(t.getTime())) {
    throw new Error('Invalid fromDate/toDate');
  }
  const fromBound = f.toISOString().split('T')[0] + 'T00:00:00Z';
  const toBound   = t.toISOString().split('T')[0] + 'T23:59:59Z';

  // Always constrain by date
  parts.push(`receivedDateTime ge ${fromBound}`);
  parts.push(`receivedDateTime le ${toBound}`);

  // Sender (exact)
  if (isNonEmpty(from)) {
    parts.push(`from/emailAddress/address eq '${escapeOData(from.trim())}'`);
  }

  // specific recipient (To/CC) if provided
  if (isNonEmpty(to)) {
    const addr = escapeOData(to.trim());
    parts.push(
        `(toRecipients/any(r:r/emailAddress/address eq '${addr}') or ` +
        `ccRecipients/any(r:r/emailAddress/address eq '${addr}'))`
    );
  }

  // Subject tolerance (exact OR prefix OR token AND-chain)
  if (isNonEmpty(subject)) {
    const subjRaw = subject.trim();
    const exact   = `subject eq '${escapeOData(subjRaw)}'`;
    const prefix  = `startswith(subject,'${escapeOData(subjRaw.slice(0, Math.min(20, subjRaw.length)))}')`;

    const rawTokens = subjRaw.split(/\s+/).filter(Boolean);
    const lower     = rawTokens.map(t => t.toLowerCase());
    const keptLower = removeStopwords(lower);
    const keepSet   = new Set(keptLower);
    const meaningfulOriginal = Array.from(
        new Set(rawTokens.filter(t => t.length >= 3 && keepSet.has(t.toLowerCase())))
    ).slice(0, 6);

    const tokenClause = meaningfulOriginal.length
      ? '(' + meaningfulOriginal.map(t => `contains(subject,'${escapeOData(t)}')`).join(' and ') + ')'
      : '';

    parts.push(tokenClause ? `(${exact} or ${prefix} or ${tokenClause})` : `(${exact} or ${prefix})`);
  }

  return '&$filter=' + parts.join(' and ');
}

function buildGsuiteQuery({ fromDate, toDate, from, subject }) {
  let q = '';
  if (isNonEmpty(from))   q += `from:${from.trim()} `;
  if (isNonEmpty(subject)) q += `subject:${subject.trim()} `;

  const f = new Date(fromDate);
  const t = new Date(toDate);
  const fStr = f.toISOString().split('T')[0];
  const tStr = new Date(t.getTime() + 24*60*60*1000).toISOString().split('T')[0]; // exclusive upper bound
  q += `after:${fStr} before:${tStr}`;
  return q;
}

function getFilter(body, type) {
  const { fromDate, toDate, from, to, subject } = body;
  if (type === 'Outlook') {
    return buildOutlookFilter({ fromDate, toDate, from, to, subject });
  }else{
    return buildGsuiteQuery({ fromDate, toDate, from, subject });
  }
}

const controller = async (req, res, next) => {
    try {
        const companyId = req.user.companyId;
        const { domainId, from, to, subject, fromDate, toDate } = req.body;
        let domain = await db.Domain.findOne({
            _id: domainId, companyId, deletedAt: null
        })
        if(!domain){
            throw new RequestError('invalid domain recieved');
        }
        if(domain.domainType == 'Gsuite' || domain.domainType == 'Outlook'){
            if(domain.isCredential){
                let filter = getFilter(req.body, domain.domainType);
                let advanceSearch =  await db.AdvanceSearch.create({
                    domainId, companyId, from, to, subject, 
                    filter, fromDate, toDate, syncStartAt: new Date()
                })
                if(domain.domainType == 'Gsuite'){
                    queue.create('process-advanced-search-gsuite', {
                        domainId, companyId, from, to, subject, filter,
                        advanceSearchId: advanceSearch._id, fromDate, toDate,
                    }).removeOnComplete(true).priority('high').save();
                } else {
                    queue.create('process-advanced-search-outlook', {
                        domainId, companyId, from, to, subject, filter,
                        advanceSearchId: advanceSearch._id, fromDate, toDate,
                    }).removeOnComplete(true).priority('high').save();
                }
            } else {
                throw new RequestError('Domain Credentials not uploaded properly please upload and try again.');    
            }
        } else {
            throw new RequestError('Domain type not define');
        }
        userActivity.create(req, 'Advanced Search', `from: ${from}, subject: ${subject}`);
        logger.info(`User ${req.user.email} proceed Advanced Search`, { client: req.user.companyId, request: req, event: 'Advanced Search' });
        return res.status(200).send({ success: true, message: 'Searching start successfully' })
    } catch (error) {
        console.log('error', error);
        next(error)
    }
}

const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), jwtStrategy, restrictFeature("TPIR-ADVS"), checkCompanyValidityStrategy, permission('AdvancedSearch','Write'), validateBody(validator), controller);
export default apiRouter;