import Joi from 'joi';
import express from 'express';


import { permission } from '../../middlewares/permission';
import { checkCompanyValidityStrategy, checkLicenseStrategy, checkTaConnectionStrategy, jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import scheduler from '../../utils/scheduler';

const validator = Joi.object({
  isActive: Joi.boolean().required(),
  domainId: Joi.string().hex().length(24).required(),
  days: Joi.alternatives().conditional('isActive', {
    is: true,
    then: Joi.array().items(Joi.string()).required(),
    otherwise: Joi.any().strip()
  }),
  hour: Joi.alternatives().conditional('isActive', {
    is: true,
    then: Joi.string().required(),
    otherwise: Joi.any().strip()
  }),
  minute: Joi.alternatives().conditional('isActive', {
    is: true,
    then: Joi.string().required(),
    otherwise: Joi.any().strip()
  }),
  timeZone: Joi.alternatives().conditional('isActive', {
    is: true,
    then: Joi.string().required(),
    otherwise: Joi.any().strip()
  }),
  syncType: Joi.alternatives().conditional('isActive', {
    is: true,
    then: Joi.string().valid('allUsers', 'specificGroups').required(),
    otherwise: Joi.any().strip()
  }),
  groupIds: Joi.alternatives().conditional('isActive', {
    is: true,
    then: Joi.array().allow(null),
    otherwise: Joi.any().strip()
  }),
  isUserSyncMirrored: Joi.alternatives().conditional('isActive', {
    is: true,
    then: Joi.boolean().required(),
    otherwise: Joi.any().strip()
  })
});


const controller = async (req, res, next) => {
    const { isActive, domainId, days = [], hour, minute, timeZone, syncType, groupIds = [], isUserSyncMirrored } = req.body;
    const companyId = req.user.companyId;
    try {
      // Persist the settings
      await db.Domain.updateOne(
        { _id: domainId, companyId },
        { isScheduledSyncing: { isActive, daysOfWeek: days, hour, minute, syncType, timeZone, groupIds, lastSync: null, isUserSyncMirrored } }
      );

      if (!isActive) {
        scheduler.cancelScheduledJob(domainId);
        return res.status(200).json({ success: true, message: 'Scheduled syncing disabled!' });
      }

      // Schedule (or re-schedule) the job
      await scheduler.scheduleDomainAutoSyncSetting({
        domainId,
        companyId,
        days,
        hour,
        minute,
        timeZone,
        syncType,
        groupIds,
        isUserSyncMirrored
      });

      return res.status(200).json({ success: true, message: 'Auto Sync scheduled successfully' });
    } catch (err) {
      next(err);
    }
}

const apiRouter = express.Router();
apiRouter.route('/').post(
    htmlsanitize(), 
    jwtStrategy, 
    checkTaConnectionStrategy, 
    checkCompanyValidityStrategy, 
    permission('Users','Write'),  
    validateBody(validator), 
    controller);

export default apiRouter;

