import express from 'express';
import { jwtStrategy } from '../../middlewares/strategy.js';
import { permission } from '../../middlewares/permission.js';

/**
 * @api {post} /api/v1/domain/clear-domain-creds Clear domain credentials
 * @apiName ClearDomainCreds
 * @apiGroup Domains
 * @apiDescription Clears credentials for a domain (Manual or Direct integration)
 *
 * @apiBody {String} domainId  The domain's ObjectId
 *
 * @apiSuccess {Boolean} success   true
 * @apiSuccess {String}  message   Credentials cleared successfully
 *
 * @apiError (400) {Boolean} success false
 * @apiError (400) {String}  message  Missing or invalid domainId
 * @apiError (404) {Boolean} success false
 * @apiError (404) {String}  message  Domain not found
 */

const controller = async (req, res, next) => {
	try {
		const { domainId } = req.body;
		const companyId = req.user.companyId;
		if (!domainId || typeof domainId !== 'string' || domainId.length !== 24) {
            throw new RequestError('Missing or invalid domainId.', 400);
		}
		const domain = await db.Domain.findOne({ _id: domainId, companyId, deletedAt: null });
		if (!domain) {
			throw new RequestError('Domain not found.', 404);
		}
		// If already cleared
		if (!domain.isCredential && (!domain.credential || Object.keys(domain.credential).length === 0)) {
			return res.status(200).json({ success: true, message: 'Credentials already cleared.' });
		}
		// Clear credentials and set isCredential to false
		domain.credential = {};
		domain.isCredential = false;
		await domain.save();
		return res.status(200).json({ success: true, message: 'Credentials cleared successfully.' });
	} catch (error) {
		console.error('Error in clear-domain-creds:', error);
		next(error);
	}
};

const apiRouter = express.Router();
apiRouter.route('/')
	.post(jwtStrategy, permission('Domain', 'Write'), controller);

export default apiRouter;
