import Joi from 'joi';
import express from 'express';

import { randomString } from '../../functions';
import userActivity from '../../service/audit-log';
import { checkCompanyValidityStrategy, checkTaConnectionStrategy, jwtStrategy } from '../../middlewares/strategy';
import { permission } from '../../middlewares/permission';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';

const validator = Joi.object().keys({
    domainName: Joi.string().regex(/^[a-zA-Z0-9][a-zA-Z0-9-]{1,61}[a-zA-Z0-9](?:\.[a-zA-Z]{2,})+$/).required(),
    domainType: Joi.any().valid('Gsuite', 'Outlook'),
    domainTypeForOutlook: Joi.any().valid('OFFICE365', 'EXCHANGE', 'null'),
    groups: Joi.array()
});

/**
 * @api {post} /api/v1/domain/domain-create domain-create
 * @apiName domain-create
 * @apiGroup Domains
 * @apiDescription create a new domain
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {String}                  domainName        Name of domain
 * @apiBody {String= Gsuite,Outlook}  domainType        Type of domain
 * 
 * @apiSuccess {Boolean} Success    true
 * @apiSuccess {String}  txtRecord  txtRecord to be added in dns for verification
 * @apiSuccess {String}  domainId   id of domain created
 * @apiSuccess {String}  message    domain added successfully
  
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *       {
 *           "success": true,
 *           "message": "Searching start successfully"
 *       }
 * 
 *  @apiError (400)  {Boolean} success-1     false
 *  @apiError (400)  {Array}   messages-1    same domain already exists try with different domain name
 *  @apiError (500)  {Boolean} success-2     false
 *  @apiError (500)  {Array}   messages-2    domainType must be one of [Gsuite, Outlook]
 * 
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 400 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *               "same domain already exists try with different domain name"
 *           ]
 *       }
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 500 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *                "\"domainType\" must be one of [Gsuite, Outlook]"
 *           ]
 *       }
 */


const controller = async (req, res, next) => {
    try {
        const { domainType, domainName, domainTypeForOutlook, groups } = req.body;
        const companyId = req.user.companyId;

        let domain = await db.Domain.findOne({
            domainName:domainName.toLowerCase(), 
            domainType,
            domainTypeForOutlook:( domainTypeForOutlook == 'null' || null) ? "null" : domainTypeForOutlook ,
            deletedAt: null
        })

        if(domainType == 'Outlook'){
            if(domain && domainTypeForOutlook == domain.domainTypeForOutlook) throw new RequestError('same domain already exists try with different domain name');
        }
        else{
            if(domain && domainType == domain.domainType) throw new RequestError('same domain already exists try with different domain name');
        }

        // if(domain){
        //     throw new RequestError('same domain already exists try with different domain name');
        // }

        // const addedDomains = await db.Domain.countDocuments({ companyId, deletedAt: null });
        // const domainLimit = await db.MetaData.findOne({ companyId, name: "domainLimit" })

        // if(!domainLimit) throw new RequestError('License Key not updated, Check License Key!');
        // if(!(+domainLimit.value > addedDomains)) throw new RequestError(`Domain Limit Exceeded! Only ${+domainLimit.value} domain(s) are granted as per your license!` ); 

        const txtRecord = `tab-site-verification=${randomString(40)}`;
        
        let isDeletedDomain = await db.Domain.findOne(
            { domainName, deletedAt: { $ne: null } },
            { _id: 1 }
          );
        if(isDeletedDomain !== null){
            domain = await db.Domain.updateOne({ _id: isDeletedDomain._id},{domainType: domainType ,deletedAt: null, isVerified:true, domainTypeForOutlook: domainTypeForOutlook, isCredential: false, credential: null, companyId })
            if(groups && groups.length >=1) groups.forEach(async (group)=> await db.SubDomainGroup.create({groupName: group, companyId, domainId: isDeletedDomain._id}) )
            domain._id = isDeletedDomain._id
        }else{ 
            if(domainType == 'Gsuite') domain = await db.Domain.create({ domainName, companyId, txtRecord, domainType, isVerified: true, domainTypeForOutlook:null})
            else domain = await db.Domain.create({ domainName, companyId, txtRecord, domainType, isVerified: true, domainTypeForOutlook, isCredential:false})
            if(groups && groups.length >=1) groups.forEach(async (group)=> await db.SubDomainGroup.create({groupName: group, companyId, domainId: domain._id}) )
        }
        
        userActivity.create(req, 'Create Domain', domainName);
        logger.info(`User ${req.user.email} proceed Create Domain`, { client: req.user.companyId, request: req, event: 'Create Domain' });

        res.status(200).json({ 
            success: true, 
            txtRecord: txtRecord,
            domainId: domain._id,
            message: 'domain added successfully'
        })

    } catch (err) {
        console.log(err);
        next(err)
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), jwtStrategy, checkCompanyValidityStrategy, permission('Domain','Write'), validateBody(validator), controller);
export default apiRouter;