import Joi from 'joi';
import path from 'path';
import multer from 'multer';
import express from 'express';
import { fromGsuite } from '../../service/credentials';
import { permission } from '../../middlewares/permission';
import { checkCompanyValidityStrategy, checkTaConnectionStrategy, jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import { scanFile } from '../../middlewares/scanAndUploadFile';


let credentialFilePath = path.join(path.dirname(require.main.filename), 'Content', 'Credentials');

let storage = multer.diskStorage({
    destination: function (req, file, cb) {
        cb(null, credentialFilePath);
    },
    filename: function (req, file, cb) {
        cb(null, Date.now() + path.extname(file.originalname));
    }
})

let uploadAttachment = multer({ storage: storage, limits: { fileSize: 10485760 } })

const validator = Joi.object().keys({
    domainId: Joi.string().hex().length(24).required(),
    adminEmail: Joi.string().email().required(),
    file: Joi.string().optional().allow(null).empty(''),
});

const controller = async (req, res, next) => {
    try {
        const { domainId, adminEmail } = req.body;
        const fileName = req.file? req.file.filename: null;
        const domain = await db.Domain.findOne({ _id: domainId, companyId: req.user.companyId, deletedAt: null});
        if (!domain) {
            throw new RequestError('Domain does not exist or wrong domain id passed!', 404);
        }
        if(!fileName) return res.status(200).send({ success: true, message: "No File Uploaded. Credentials not updated!" })    
        if (!domain.isVerified) throw new RequestError('Please verify the domain first', 400);

        let credential = {
            adminEmail,
            fileName: fileName? fileName: domain.fileName
        }
        try {
            const {service, fileName} = await fromGsuite(credential);
            if (!service && !fileName) {
                throw new RequestError('Invalid Credentials - Unable to authorize', 400);
            }
        } catch (error) {
            throw new RequestError('Invalid Credentials - Unable to authorize', 400);
        }

        await db.Domain.findOneAndUpdate({ _id: domainId, companyId: req.user.companyId }, { 
            credential, isCredential: true
        });
        return res.status(200).send({ success: true, message: "Credentials updated successfully" })
    } catch (err) {
        console.log(err);
        next(err)
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), jwtStrategy, checkCompanyValidityStrategy, permission('Domain', 'Write'), uploadAttachment.single('file'),  scanFile, validateBody(validator), controller);
export default apiRouter;