import Joi from 'joi';
import express from 'express';

import crypto from '../../crypto';
import { permission } from '../../middlewares/permission';
import { checkCompanyValidityStrategy, checkTaConnectionStrategy, jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import getToken from '../../service/get-outlook-token';

const validator = Joi.object().keys({
    tenantId: Joi.string().required(),
    clientId: Joi.string().required(),
    clientSecret: Joi.string().required(),
    domainId: Joi.string().hex().length(24).required(),
});

/**
 * @api {post} /api/v1/domain/domain-credential-outlook
 * @apiName domain-credential-outlook
 * @apiGroup Domains
 * @apiDescription Upload Outlook Credentials
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody   {String}  domainId         id of the domain
 * @apiBody   {String}  tenantId         tenantId of the domain
 * @apiBody   {String}  clientId         clientId of the domain
 * @apiBody   {String}  clientSecret     clientSecret of the domain
 * 
 * @apiSuccess {Boolean} Success    true
 * @apiSuccess {String}  message    Credentials updated successfully
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *       {
 *           "success": true,
 *           "message": "Credentials updated successfully"
 *       }
 *  
 *  @apiError (500)  {Boolean}  success    false
 *  @apiError (500)  {Array}    messages   Invalid domain recieved
 * 
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 500 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *                "Invalid domain recieved"
 *           ]
 *       }
 */


const controller = async (req, res, next) => {
    try {
        const { domainId, clientId, clientSecret, tenantId } = req.body;
        const companyId = req.user.companyId;

        const domain = await db.Domain.findOne({
            _id: domainId, companyId, deletedAt: null
        })
        if (!domain) {
            throw new RequestError('Domain does not exist or wrong domain id passed!', 404);
        }
        if (!domain.isVerified) {
            throw new RequestError('Please verify the domain first', 400);
        }
        const credential = {
            clientId: crypto.encrypt(clientId),
            tenantId: crypto.encrypt(tenantId),
            clientSecret: crypto.encrypt(clientSecret),
            integrationType: 'Manual'
        };
        // validate credentials by trying to get access token
        try {
            const accessToken = await getToken.getOutlookAuthToken(tenantId, clientId, clientSecret);
            if (!accessToken || !accessToken.access_token) {
                throw new RequestError('Invalid Credentials - Unable to authorize', 400);
            }
        } catch (error) {
            throw new RequestError('Invalid Credentials - Unable to authorize', 400);
        }

        await db.Domain.findOneAndUpdate(
            { _id: domainId, companyId },
            { credential, isCredential: true }
        );

        return res.status(200).send({ success: true, message: "Credentials updated successfully" });

    } catch (err) {
        console.log(err);
        next(err)
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), jwtStrategy, checkCompanyValidityStrategy, permission('Domain','Write'), validateBody(validator), controller);
export default apiRouter;