import Joi from 'joi';
import express from 'express';
import schedule from "node-schedule";
import { checkTaConnectionStrategy, jwtStrategy } from '../../middlewares/strategy';
import { permission } from '../../middlewares/permission';
import { validateBody } from '../../middlewares/validator';

const validator = Joi.object().keys({
    domainId: Joi.string().hex().length(24).required()
});

/**
 * @api {post} /api/v1/domain/domain-delete?domainId=63b676c744ed3b8a72386597 domain-delete
 * @apiName domain-delete
 * @apiGroup Domains
 * @apiDescription Delete a domain
 * 
 * @apiVersion 1.0.0
 * 
 * @apiQuery   {String}  domainId   id of the domain
 * 
 * @apiSuccess {Boolean} Success  true
 * @apiSuccess {String}  message  Domain Deleted Successfully.
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *       {
 *           "success": true,
 *           "message": "Domain Deleted successfully"
 *       }
 *  
 *  @apiError (500)  {Boolean}  success    false
 *  @apiError (500)  {Array}    messages   Domain Does not Exists
 * 
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 500 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *                "Domain Does not Exists"
 *           ]
 *       }
 */


const controller = async (req, res, next) => {
    try {
        const { domainId } = req.query;
        const companyId = req.user.companyId;

        const domain = await db.Domain.findOne({
            _id: domainId,
            companyId,
            deletedAt: null
        })
        if(!domain) {
            throw new RequestError('Domain Does not Exists');
        }
        // removing already scheduled job
        const removeAutoSync = schedule.cancelJob(domainId);
        if (removeAutoSync) {
            console.log(`Job with ID ${domainId} cancelled successfully.`);
        } else {
            console.log(`No job found with ID ${domainId}.`);
        }

        domain.deletedAt = new Date();
        domain.isSyncingUser = false
        domain.isSyncingUserError = false
        domain.isSyncingUserErrorFile = null
        domain.isGroupSyncEnabled = false
        domain.groups = []
        domain.subDomaingroups = []
        domain.isScheduledSyncing = { isActive: false, daysOfWeek: [], hour: null, minute: null, syncType: "All Users", timeZone: null, groupIds: [], lastSync: null}
        domain.lastAllUsersSyncAt = null
        await domain.save();

        await db.SubDomainGroup.deleteMany({
            domainId, companyId
        })

        await db.UserGroup.deleteMany({
            domainId, companyId
        })
        
        await db.User.deleteMany({
            domainId: domainId,
            deletedAt: null
        })

        const groupsOfCompany = await db.Group.find({
            deletedAt: null,
            companyId,
            isDefaultGroup: false,
            groupName:{$ne:"Individual-Users"}
        });
        
        if (groupsOfCompany.length > 0) {
            // Getting all group IDs
            const groupIds = groupsOfCompany.map(group => group._id);
        
            // Getting user counts for each group in a single query
            const userCounts = await db.User.aggregate([
                { $match: { companyId, deletedAt: null, groups: { $in: groupIds } } },
                { $unwind: "$groups" },
                { $match: { groups: { $in: groupIds } } },
                { $group: { _id: "$groups", count: { $sum: 1 } } },
            ]);
        
            // Creating a map of group ID to user count
            const userCountMap = userCounts.reduce((acc, curr) => {
                acc[curr._id.toString()] = curr.count;
                return acc;
            }, {});
        
            // Finding groups with zero users and delete them
            const groupsToDelete = groupsOfCompany
                .filter(group => !userCountMap[group._id.toString()])
                .map(group => group._id);
        
            if (groupsToDelete.length > 0) {
                await db.Group.deleteMany({ _id: { $in: groupsToDelete }, companyId });
            }
        }
        

        return res.status(200).send({ succees: true, message: "Domain Deleted Successfully."})


    } catch (error) {
        console.log(error);
        next(error)
    }
}

const apiRouter = express.Router();
apiRouter.route('/').delete(jwtStrategy, checkTaConnectionStrategy, permission('Domain','Delete'), validateBody(validator), controller);
export default apiRouter;