import Joi from 'joi';
import dns from 'node:dns';
import express from 'express';

import { permission } from '../../middlewares/permission';
import { checkCompanyValidityStrategy, checkTaConnectionStrategy, jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';

const validator = Joi.object().keys({
    domainId: Joi.string().hex().length(24).required()
});


/**
 * @api {post} /api/v1/domain/domain-verify?domainId=63b676c744ed3b8a72386597 domain-verify
 * @apiName domain-verify
 * @apiGroup Domains
 * @apiDescription verify a new domain
 * 
 * @apiVersion 1.0.0
 * 
 * @apiQuery {String}    domainId   id of the domain
 * 
 * @apiSuccess {Boolean} Success-1                        true
 * @apiSuccess {String}  message-1                        Domain Verified successfully
 * @apiSuccess {Boolean} Success-2                        true
 * @apiSuccess {String}  message-2                        Domain already Verified!
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *       {
 *           "success": true,
 *           "message": "Domain Verified successfully"
 *       }
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *       {
 *           "success": true,
 *           "message": "Domain already Verified!"
 *       }
 *  
 *  @apiError (400)  {Boolean}  success    false
 *  @apiError (400)  {Array}    messages   TXT record not updated on DNS, Please update TXT record!
 *  @apiError (404)  {Boolean}  success    false
 *  @apiError (404)  {Array}    messages   Domain does not exist or wrong domain id passed!
 * 
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 400 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *               "TXT record not updated on DNS, Please update TXT record!"
 *           ]
 *       }
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 500 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *                "Domain does not exist or wrong domain id passed!"
 *           ]
 *       }
 */

const checkDnsRecord = async(domain, txtRecord) => {
    return new Promise((resolve, reject) => {
        dns.resolveTxt(domain, (err, record) => {
            if (err) {
                resolve(false)
            } else {
                let status = false;
                for(let i=0; i<record.length; i++){
                    if(record[i][0] == txtRecord){
                        status = true;
                    }
                }
                resolve(status)
            }
        })
    })
}
const controller = async (req, res, next) => {
    try {
        const { domainId } = req.query;
        const companyId = req.user.companyId;
        const domain = await db.Domain.findOne({
            _id: domainId, 
            companyId,
            deletedAt: null
        })
        if (!domain) {
            throw new RequestError('Domain does not exist or wrong domain id passed!', 404);
        }
        if (domain && !domain.isVerified) {
            let checkTxt = await checkDnsRecord(domain.domainName, domain.txtRecord)
            if (!checkTxt) {
                throw new RequestError('TXT record not updated on DNS, Please update TXT record!')
            }
            await db.Domain.findOneAndUpdate({ _id: domainId, companyId }, { isVerified: true });
            return res.status(200).send({ success: true, message: "Domain Verified successfully" })
        }
        return res.status(200).send({ success: true, message: "Domain already Verified!"});
    } catch (err) {
        console.log(err);
        next(err)
    }
}
const apiRouter = express.Router();
apiRouter.route('/').get(jwtStrategy, checkCompanyValidityStrategy, permission('Domain','Write'), validateBody(validator), controller);
export default apiRouter;