import express from 'express';
import { jwtStrategy } from '../../middlewares/strategy.js';
import { permission } from '../../middlewares/permission.js';
import crypto from '../../crypto.js';

/**
 * @api {get} /api/v1/domain/get-auth-type?domainId=... Get domain integration type
 * @apiName GetDomainAuthType
 * @apiGroup Domains
 * @apiDescription Returns the integrationType (Manual or Direct) for a domain
 *
 * @apiQuery {String} domainId  The domain's ObjectId
 *
 * @apiSuccess {Boolean} success   true
 * @apiSuccess {String}  integrationType  'Manual' or 'Direct' or null
 *
 * @apiError (400) {Boolean} success false
 * @apiError (400) {String}  message  Missing or invalid domainId
 * @apiError (404) {Boolean} success false
 * @apiError (404) {String}  message  Domain not found
 */

const controller = async (req, res, next) => {
	try {
		const { domainId } = req.query;
		const companyId = req.user.companyId;
		if (!domainId || typeof domainId !== 'string' || domainId.length !== 24) {
			throw new RequestError('Missing or invalid domainId.', 400);
		}
		const domain = await db.Domain.findOne({ _id: domainId, companyId, deletedAt: null });
		if (!domain) {
			throw new RequestError('Domain not found.', 404);
		}
		let integrationType = null;
		let credentials = null;
		if (domain.credential && domain.credential.integrationType) {
			integrationType = domain.credential.integrationType;
			// Prepare credentials object based on integrationType
			if (integrationType === 'Manual') {
				credentials = {};
				if (domain.credential.clientId) {
					credentials.clientId = crypto.decrypt(domain.credential.clientId);
				}
				if (domain.credential.tenantId) {
					credentials.tenantId = crypto.decrypt(domain.credential.tenantId);
				}
				if (domain.credential.clientSecret) {
					const decryptedSecret = crypto.decrypt(domain.credential.clientSecret);
					credentials.clientSecret =
						decryptedSecret && decryptedSecret.length > 5
							? decryptedSecret.slice(0, 5) + '*'.repeat(decryptedSecret.length - 5)
							: '*'.repeat(decryptedSecret ? decryptedSecret.length : 0);
				}
			} else if (integrationType === 'Direct') {
				credentials = {};
				if (domain.credential.tenantId) {
					credentials.tenantId = crypto.decrypt(domain.credential.tenantId);
				}
			}
		}
		return res.status(200).json({ success: true, integrationType, credentials });
	} catch (error) {
		console.error('Error in get-auth-type:', error);
		next(error);
	}
};

const apiRouter = express.Router();
apiRouter.route('/')
	.get(jwtStrategy, permission('Domain', 'Read'), controller);

export default apiRouter;
