import express from 'express';

import { jwtStrategy } from '../../middlewares/strategy';
import { permission } from '../../middlewares/permission';

/**
 * @api {get} /api/v1/domain/domain-list domain-list
 * @apiName list
 * @apiGroup Domains
 * @apiDescription Returns a list of all domains of a user's company with active and total users for domain's page
 * 
 * @apiVersion 1.0.0
 * 
 * @apiSuccess {Boolean} Success    true
 * @apiSuccess {Array}   domains    list of all domains
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *       {    
 *         "success": true,
 *         "domains": [
 *             {
 *                 "_id": "63d0eaaa032f943fc31e4ea2",
 *                 "domainName": "kdmarc.com",
 *                 "txtRecord": "tab-site-verification=W7CIoMg0Bitq6hahLZ4pW8aL6h4I23DLTHV88cFj",
 *                 "isVerified": true,
 *                 "companyId": "63d0ea61032f943fc31e4e80",
 *                 "domainType": "Outlook",
 *                 "isCredential": true,
 *                 "isSyncingUser": false,
 *                 "deletedAt": null,
 *                 "createdAt": "2023-01-25T08:39:06.842Z",
 *                 "updatedAt": "2023-01-25T09:00:27.078Z",
 *                 "__v": 0,
 *                 "syncStartAt": "2023-01-25T09:00:25.226Z",
 *                 "syncEndAt": "2023-01-25T09:00:27.077Z",
 *                 "totalUser": 46,
 *                 "activeUser": 46
 *             },
 *             {
 *                 "_id": "63d0f49e4b01a8cf9308bf00",
 *                 "domainName": "kratikal.com",
 *                 "txtRecord": "tab-site-verification=J6oFlCGhQzEWZKVgHqQGywiru4gslcxDgCBEWWrQ",
 *                 "isVerified": false,
 *                 "companyId": "63d0ea61032f943fc31e4e80",
 *                 "domainType": "Outlook",
 *                 "isCredential": false,
 *                 "isSyncingUser": false,
 *                 "deletedAt": null,
 *                 "createdAt": "2023-01-25T09:21:34.965Z",
 *                 "updatedAt": "2023-01-25T09:21:34.965Z",
 *                 "__v": 0,
 *                 "totalUser": 0,
 *                 "activeUser": 0
 *             }
 *         ]
 *      }
 * 
 */

const controller = async (req, res, next) => {
    try {
        const companyId = req.user.companyId;

        let domains = await db.Domain.find({
            companyId,
            deletedAt: null
        }).select('-credential');
        for(let i=0; i<domains.length; i++){
            let totalUser = await db.User.countDocuments({
                domainId: domains[i]._doc._id,
                deletedAt: null
            })
            let activeUser = await db.User.countDocuments({
                domainId: domains[i]._doc._id, isActive: true, deletedAt: null
            });

            let subDomainGroups = await db.SubDomainGroup.find({companyId, domainId: domains[i]._doc._id,})
            let userGroups = await db.UserGroup.find({companyId, domainId: domains[i]._doc._id},{_id:0 , label:1, value:1})

            domains[i] = { ...domains[i]._doc, totalUser, activeUser, subDomainGroups, userGroups };
        }
        
        
       
        res.status(200).json({ success: true, domains });
    } catch (err) {
        console.log(err);
        next(err)
    }
}

const apiRouter = express.Router();
apiRouter.route('/').get(jwtStrategy, permission('Domain','Read'), controller);
export default apiRouter;