import Joi from 'joi';
import express from 'express';

import mailer from '../../mailer';
import crypto from '../../crypto';
import sanitizeHtml from 'sanitize-html';

import { randomString } from '../../functions';
import userActivity from '../../service/audit-log';
import { sanitize } from '../../middlewares/sanitizer';
import { checkCompanyValidityStrategy, checkTaConnectionStrategy, jwtStrategy } from '../../middlewares/strategy';
import { permission } from '../../middlewares/permission';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import rateLimit from 'express-rate-limit';


let validator = Joi.object().keys({
    firstName: Joi.string().required(),
    lastName: Joi.string().allow(null, '').required(),    
    email: Joi.string().email().required(),
});

const strictLimiter = rateLimit({
    windowMs: 1 * 60 * 1000, // 1 minute
    max: 5, // 5 requests per minute
    message: "Too many requests to this endpoint, please try again later.",
  });

/**
 * @api {post} /api/v1/employee/add add
 * @apiName add
 * @apiGroup Employee
 * @apiDescription Add an IAM Employee
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody   {String}  email        email of the employee
 * @apiBody   {String}  name         name of the employee
 * 
 * @apiSuccess {Boolean} Success    true
 * @apiSuccess {String}  message    IAM user added successfully and set password link has been send to the IAM user email
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 201 OK
 *       {
 *           "success": true,
 *           "message": "IAM user added successfully and set password link has been send to the IAM user email"
 *       }
 *  
 *  @apiError (500)  {Boolean}  success    false
 *  @apiError (500)  {Array}    messages   Same email already exists please try with another email!
 * 
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 500 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *                "Same email already exists please try with another email!"
 *           ]
 *       }
 */

let controller = async (req, res, next) => {
    try {
        let { email, firstName, lastName } = req.body;
        email = sanitizeHtml(email)
        firstName = sanitizeHtml(firstName)
        lastName = sanitizeHtml(lastName)

        if(!email.length) throw new RequestError("Email is empty after sanitizing!")
        else if(!firstName.length) throw new RequestError("First Name is empty after sanitizing!")

        const isHyperLink = /^[a-zA-Z]+$/

        if(!isHyperLink.test(firstName)) throw new RequestError("First name cannot be a hyperlink!")

        const companyId = req.user.companyId;
        const verfKey = randomString(10);

        const checkEmployee = await db.Employee.findOne({ 
            email,
            companyId, 
            deletedAt: null
        })
        if (checkEmployee) {
            throw new RequestError("Same email already exists please try with another email!");
        }
        let encryptEmail = await crypto.encrypt(email);
        encryptEmail = encodeURIComponent(encryptEmail);
        let newEmployee = await db.Employee.create({ 
            email, firstName, lastName, companyId, verfKey, verfKeyExpiry: new Date()
        })
        
        userActivity.create(req, 'Add IAM User', email);
        logger.info(`User ${req.user.email} Add IAM User`, { client: req.user.companyId, request: req, event: 'Add IAM User' });

        await mailer.sendEmployeePassword(firstName ,email, encryptEmail, verfKey, companyId);
        return res.status(201).send({ success: true, message: "IAM user added successfully and set password link has been send to the IAM user email", employeeId: newEmployee._id})

    } catch (err) {
        console.log(err);
        next(err)
    }
}

const apiRouter = express.Router();
apiRouter.route('/').post( strictLimiter, htmlsanitize(), validateBody(validator), jwtStrategy, checkTaConnectionStrategy, checkCompanyValidityStrategy, permission('IAM','Write'), controller);
export default apiRouter;