import Joi from 'joi';
import express from 'express';

import { permission } from '../../middlewares/permission';
import { checkTaConnectionStrategy, jwtStrategy} from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import userActivity from '../../service/audit-log';

let validator = Joi.object().keys({
    employeeId: Joi.string().hex().length(24).required()
});

/**
 * @api {delete} /api/v1/employee/delete?employeeId=63b676c744ed3b8a72386597 delete
 * @apiName deltete
 * @apiGroup Employee
 * @apiDescription Delete an IAM Employee
 * 
 * @apiVersion 1.0.0
 * 
 * @apiQuery   {String}  employeeId   id of the employee
 * 
 * @apiSuccess {Boolean} Success      true
 * @apiSuccess {String}  message      IAM User deleted successfully
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 201 OK
 *       {
 *           "success": true,
 *           "message": "IAM User deleted successfully"
 *       }
 *  
 *  @apiError (500)  {Boolean}  success    false
 *  @apiError (500)  {Array}    messages   Same email already exists please try with another email!
 * 
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 500 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *                "Same email already exists please try with another email!"
 *           ]
 *       }
 */

let controller = async (req, res, next) => {
    try {
        const { employeeId } = req.query;
        const companyId = req.user.companyId;
        const employee = await db.Employee.findOne({ 
            _id: employeeId, 
            deletedAt: null,
            companyId 
        });
        if(!employee){
            throw new RequestError('Employee not Found', 404);
        }
       
        await employee.deleteOne({_id:employee._id});
        await db.EmployeeRole.deleteOne({employeeId});
        
        userActivity.create(req, 'Delete IAM User', employee.email);
        logger.info(`User ${req.user.email} Delete IAM User`, { client: req.user.companyId, request: req, event: 'Delete IAM User' });
        return res.status(200).send({ success:true, message: 'IAM User deleted successfully' });
    } catch (error) {  
        console.log(error);      
        return next(error)
    }
}
const apiRouter = express.Router();
apiRouter.route('/').delete( jwtStrategy, checkTaConnectionStrategy, permission('IAM','Delete'), validateBody(validator), controller);
export default apiRouter;