import express from 'express';

import { permission } from '../../middlewares/permission';
import { jwtStrategy, checkCompanyValidityStrategy } from '../../middlewares/strategy';

/**
 * @api {get} /api/v1/employee/list list
 * @apiName list
 * @apiGroup Employee
 * @apiDescription List all IAM Employees
 * 
 * @apiVersion 1.0.0
 * 
 * @apiSuccess {Boolean}  Success      true
 * @apiSuccess {Array}    employees    list of all employees
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 201 OK
 *      {
 *    "success": true,
 *    "employees": [
 *        {
 *            "_id": "63d0f1c078b9830aa47b3052",
 *            "name": "Arpit Rao",
 *            "email": "arpit.rao@kratikal.com",
 *            "isActive": false,
 *            "isVerify": false
 *        },
 *        {
 *            "_id": "63d75011f111d4088dd478e2",
 *            "name": "sarvesh.patel",
 *            "email": "sarvesh.patel@kratikal.com",
 *            "isActive": false,
 *            "isVerify": false
 *        }
 *  
 *  
 */

let controller = async (req, res, next) => {
    try {
        
        const companyId = req.user.companyId;
        console.log({ companyId });

        // Step 1: Find employees for the given company
        const employees = await db.Employee.find({
            companyId,
            deletedAt: null
        }).select('firstName lastName email isActive isVerify');

        // Step 2: Get employee IDs
        const employeeIds = employees.map(employee => employee._id);

        // Step 3: Find employee roles for the given employee IDs and company
        const employeeRoles = await db.EmployeeRole.find({
            employeeId: { $in: employeeIds },
            companyId,
            deletedAt: null
        })
        .populate({
            path: 'employeeId',
            match: { deletedAt: null },
            select: 'firstName lastName email isActive isVerify'
        })
        .populate({
            path: 'roleIds',
            populate: {
                path: 'policyIds',
                select: 'name'
            }
        });

        // Step 4: Prepare final response
        return res.status(200).send({ success: true, employees: employeeRoles });

    } catch (error) {        
        return next(new RequestError(error))
    }
}
const apiRouter = express.Router();
apiRouter.route('/').get(  jwtStrategy, permission('IAM','Read'), controller);
export default apiRouter;