import express from 'express';
import config from '../../config/index.js';
import getToken from '../../service/get-outlook-token.js';
import crypto from '../../crypto.js';

// This API is for domain OAuth configuration via Azure callback (Direct integration)
const controller = async (req, res, next) => {
	const redirectUrl = `${config.app.url}/domains`;
	try {
		const { tenant: tenantId, admin_consent, state: encodedMetaData } = req.query;
		console.log('Received query parameters:', { tenantId, admin_consent, encodedMetaData });
		if (admin_consent?.toLowerCase() !== 'true') {
			throw new RequestError('Admin consent was not granted.');
		}

		if (!tenantId || !encodedMetaData) {
			throw new RequestError('Invalid callback request. Required parameters are missing.');
		}

		let decodedData;
		try {
			decodedData = JSON.parse(Buffer.from(encodedMetaData, 'base64').toString('utf-8'));
		} catch (e) {
			throw new RequestError('Invalid state data. Unable to decode.');
		}
		const { companyId, domainId } = decodedData;
		console.log('Decoded metadata:', { companyId, domainId });
		if (!companyId || !domainId) {
			throw new RequestError('Invalid state data. Company or domain information is missing.');
		}

		// Find the domain for the company
		const domain = await db.Domain.findOne({ _id: domainId, companyId, deletedAt: null });
		if (!domain) {
			const errorMessage = 'Domain not found for this company.';
			return res.redirect(`${redirectUrl}?success=false&message=${encodeURIComponent(errorMessage)}`);
		}
		if (!domain.isVerified) {
			const errorMessage = 'Please verify the domain first.';
			return res.redirect(`${redirectUrl}?success=false&message=${encodeURIComponent(errorMessage)}`);
		}
		// If credentials already exist and integrationType is Direct, block
		if (domain.isCredential && domain.credential && domain.credential.tenantId && domain.credential.integrationType === 'Direct') {
			const errorMessage = 'Credentials already set for this domain.';
			return res.redirect(`${redirectUrl}?success=false&message=${encodeURIComponent(errorMessage)}`);
		}
		// If credentials exist for other integrationType, warn
		if (domain.isCredential && domain.credential && domain.credential.integrationType && domain.credential.integrationType !== 'Direct') {
			const errorMessage = 'Domain already has credentials for another integration type.';
			return res.redirect(`${redirectUrl}?success=false&message=${encodeURIComponent(errorMessage)}`);
		}

		// Validate Azure token using registered app credentials from config
		const clientId = config.azure.defaultDomainAppClientId;
		const clientSecret = config.azure.defaultDomainAppClientSecret;
		const tokenResponse = await getToken.getOutlookAuthToken(tenantId, clientId, clientSecret);
		if (!tokenResponse || !tokenResponse.access_token) {
			const errorMessage = 'Invalid credentials - Unable to authorize with Microsoft Azure.';
			return res.redirect(`${redirectUrl}?success=false&message=${encodeURIComponent(errorMessage)}`);
		}
		console.log('Successfully obtained access token from Azure.', tokenResponse);

		// Save only encrypted tenantId in credential, set isCredential true, set integrationType
		domain.credential = {
			tenantId: crypto.encrypt(tenantId),
			integrationType: 'Direct',
		};
		domain.isCredential = true;
		await domain.save();

		return res.redirect(`${redirectUrl}?success=true&message=Successfully authorized the application.`);
	} catch (error) {
		console.error('Error during Azure domain consent callback:', error);
		const errorMessage = error.message || 'An internal server error occurred. Please try again later.';
		return res.redirect(`${redirectUrl}?success=false&message=${encodeURIComponent(errorMessage)}`);
	}
};

const apiRouter = express.Router();
apiRouter.route('/').get(controller);

export default apiRouter;
