import express from 'express';
import Joi from 'joi';
import { jwtStrategy, checkCompanyValidityStrategy } from '../../middlewares/strategy.js';
import { permission } from '../../middlewares/permission.js';
import { validateBody } from '../../middlewares/validator.js';
import { htmlsanitize } from '../../middlewares/sanitizeHtml.js';
import config from '../../config';

// Body validator
const validator = Joi.object().keys({
    domainId: Joi.string().hex().length(24).required(),
});

/**
 * @api {post} /api/v1/integration/azure-consent-url Generate Azure consent URL for a Domain
 * @apiName GenerateAzureConsentUrlForDomain
 * @apiGroup Domains
 * @apiDescription Generates Microsoft admin consent URL for domain level Outlook integration.
 *
 * @apiBody {String} domainId Domain Id (24 hex chars).
 *
 * @apiSuccess {Boolean} success true
 * @apiSuccess {String}  message Success message
 * @apiSuccess {Object}  data Data object
 * @apiSuccess {String}  data.consentUrl Generated consent URL
 *
 * @apiError (400) {String} message Domain not verified / already has credentials
 * @apiError (404) {String} message Domain not found
 * @apiError (500) {String} message Internal server error
 */
const controller = async (req, res, next) => {
    const { companyId } = req.user;
    const { domainId } = req.body;
    try {
        // Fetch domain
        const domain = await db.Domain.findOne({ _id: domainId, companyId, deletedAt: null });
        if (!domain) {
            throw new RequestError('Domain does not exist or wrong domain id passed!', 404);
        }
        if (!domain.isVerified) {
            throw new RequestError('Please verify the domain first', 400);
        }
        if (domain.isCredential || (domain.credential && Object.keys(domain.credential || {}).length > 0)) {
            throw new RequestError('Credentials already configured for this domain', 400);
        }

        // Use domain specific default Azure App Client Id
        const defaultClientId = config.azure.defaultDomainAppClientId;
        if (!defaultClientId) {
            throw new RequestError("Default Azure domain app's client ID is not set in environment variables.");
        }

        const meta = { companyId: companyId.toString(), domainId: domainId.toString() };
        const encodedData = Buffer.from(JSON.stringify(meta)).toString('base64');
        const redirectUrl = `${config.azure.callbackDomain}/api/v1/integration/azure-callback`;
        const hintedDomain = domain.domainName || '';
        const consentUrl = config.azure.consentUrl
            .replaceAll('{{clientId}}', defaultClientId)
            .replaceAll('{{endcodedRedirectUrl}}', encodeURIComponent(redirectUrl))
            .replaceAll('{{encodedMetaData}}', encodedData)
            .replaceAll('{{hinted_domain}}', hintedDomain);

        return res.status(200).json({
            success: true,
            message: 'Consent URL generated successfully',
            data: { consentUrl },
        });
    } catch (error) {
        next(error);
    }
};

const apiRouter = express.Router();
apiRouter.route('/')
    .post(htmlsanitize(), jwtStrategy, checkCompanyValidityStrategy, permission('Domain','Write'), validateBody(validator), controller);

export default apiRouter;