import Joi from 'joi';
import express from 'express';
import { jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { permission } from '../../middlewares/permission';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import auditLog from '../../service/audit-log';

/**
 * @api {post} /api/v1/mail-actions/bulk Bulk Mail Actions
 * @apiName mail-actions-bulk
 * @apiGroup Mail-Actions
 * @apiDescription
 * Bulk mail actions including ignore operation
 *
 * @apiVersion 1.0.0
 * @apiBody {String[]} ids  Array of reportedMailIds (max 100)
 * @apiBody {String='ignore'} action  Action to perform
 *
 * @apiSuccess {Boolean} success true
 * @apiSuccess {Number} count  Number of mails processed
 * @apiSuccess {String} message  Human-friendly message
 */
const validator = Joi.object({
  ids: Joi.array().items(Joi.string().hex().length(24)).min(1).max(100).required(),
  action: Joi.string().valid('ignore').required()
});

const controller = async (req, res, next) => {
  try {
    const { ids, action } = req.body;
    const { companyId } = req.user;

    if (action === 'ignore') {
      try { 
        console.log(`[BULK][IGNORE] companyId=${companyId} requested=${ids.length}`); 
      } catch(_l){}
      
      // Only mails with no actions can be ignored
      const condition = {
        _id: { $in: ids },
        companyId,
        isTrashedEmail: false,
        isRecoveredEmail: false,
        isDeletedEmail: false,
      };
      
      const now = new Date();
      const result = await db.ReportedMail.updateMany(condition, { 
        isIgnored: true, 
        ignoredAt: now, 
        ignoredBy: req.user._id 
      });
      
      const affected = result.modifiedCount || result.nModified || 0;
      
      auditLog.create(req, 'Bulk:Ignore', { 
        count: affected, 
        requested: ids.length, 
        ids 
      }, null);
      
      const message = affected === 1 ? 'Successfully ignored the mail' : `Successfully ignored ${affected} mails`;
      
      try { 
        console.log(`[BULK][IGNORE][DONE] companyId=${companyId} affected=${affected}`); 
      } catch(_l){}
      
      return res.status(200).json({ 
        success: true, 
        message, 
        count: affected 
      });
    }

    // If other actions are added in the future, they would go here
    return res.status(400).json({ 
      success: false, 
      error: 'Unsupported action' 
    });

  } catch (err) {
    next(err);
  }
};

const apiRouter = express.Router();
apiRouter.route('/')
  .post(htmlsanitize(), jwtStrategy, permission('Emails','Delete'), validateBody(validator), controller);

export default apiRouter;