import Joi from 'joi';
import express from 'express';

import credentials from '../../service/credentials';
import { sanitize } from '../../middlewares/sanitizer';
import { checkCompanyValidityStrategy, jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import moveEmailGsuite from '../../service/move-email-gsuite';
import moveEmailOutlook from '../../service/move-email-outlook';
import mailer from '../../mailer';
import { permission } from '../../middlewares/permission';
import { queue } from '../../kue'
import { updateReportedMailCache } from '../../functions';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import updateMessageId from '../../service/update-messageId';
import restrictFeature from '../../middlewares/featureBlocker';

let validator = Joi.object().keys({
    whoElseIds: Joi.array().items(
        Joi.string().hex().length(24)                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       
    ).optional(),
    reportedMailId: Joi.string().hex().length(24).required()
});

/**
 * @api {post} /api/v1/mail-actions/move-to-trash Trash Mail
 * @apiName move-to-trash
 * @apiGroup Who-Else
 * @apiDescription Trash a particular who-else mail
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody   {String}  whoElseIds   Required who-else id of the mail to be trashed
 * 
 * @apiSuccess {Boolean} Success    true
 * @apiSuccess {String}  Message    "Successfully trashed and will be update in sometime"
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *       {
 *           "success": true,
 *           "Message": "Successfully deleted and will be update in sometime"
 *       }
 *  
 *  @apiError (500)  {Boolean}  success    false
 *  @apiError (500)  {Array}    messages    ["\"whoElseIds[0]\" is not allowed to be empty"]
 * 
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 500 Not Found
 *      {
 *          "success": false,
 *          "messages": [
 *              "\"whoElseIds[0]\" is not allowed to be empty"
 *          ]
 *      }
 */

let controller = async (req, res, next) => {
    try {
        const { reportedMailId } = req.body;
        const companyId = req.user.companyId;
        const emailReported = await db.ReportedMail.findOne({
            _id: reportedMailId,
            companyId
        })

        if(!emailReported){
            throw new RequestError('invalid reportedMailId recieved');
        }
        if(emailReported.isWhoElseProcessing=='true') throw new RequestError('Who-Else Search in Progress. Please wait and refresh before retrying.'); 
        
        let whoElseIds = await db.WhoElse.find({reportedMailId, companyId, isDeleted: false, deletedAt: null}).select('_id')        
        whoElseIds = whoElseIds.map(whoElse => whoElse._id)

        
        if(emailReported.mailType == 'Exchange'){
            const user = await db.User.findOne({ email: emailReported.to })
            mailer.sendActionResultMail(user.name, emailReported.to, emailReported.from, emailReported.subject, emailReported.mailRecievedTime, "THREAT", false, "", companyId)      
            if(emailReported.reporters.length!=0 ){
                emailReported.reporters.forEach(async (reporter)=>{
                    const user = await db.User.findOne({ email: reporter?.email })
                    mailer.sendActionResultMail(user.name, reporter.email, emailReported.from, emailReported.subject, emailReported.mailRecievedTime,"THREAT", false, "", companyId) 
                })
            }
            updateReportedMailCache(reportedMailId,{isTrashedEmail:"false", isRecoveredEmail:"false", isDeletedEmail:"true"})

            emailReported.isRecoveredEmail = false;
            emailReported.isTrashedEmail = false;
            emailReported.isDeletedEmail = true;
            await emailReported.save()
            return res.status(200).send({ success: true, Message: "Successfully marked as threat!"})
        }
        else {
            let whoElseIdsDomainWise = {};

            let whoElses = []
            var gsuiteIds = []
            
            for(const whoElse of whoElseIds){
                let result = await db.WhoElse.findOne({companyId, _id: whoElse}).populate('domainId' ,'domainType _id')
                if(result.domainId.domainType == 'Gsuite') gsuiteIds.push(result)
                else whoElses.push(result)
            }

            for(let i=0;i<whoElses.length;i++){
                let whoElse = whoElses[i]
                if(!whoElseIdsDomainWise[whoElse.domainId._id])  whoElseIdsDomainWise[whoElse.domainId._id] = []

                
                if(whoElseIdsDomainWise[whoElse.domainId._id].length > 0){
                    let lastArray = whoElseIdsDomainWise[whoElse.domainId._id].pop()
                    if(lastArray.length < 20) lastArray.push(whoElse), whoElseIdsDomainWise[whoElse.domainId._id].push(lastArray)
                    else whoElseIdsDomainWise[whoElse.domainId._id].push(lastArray), whoElseIdsDomainWise[whoElse.domainId._id].push([whoElse])
                }
                else whoElseIdsDomainWise[whoElse.domainId._id].push([whoElse])
            }      

            
            for(const domainWhoelse in whoElseIdsDomainWise){
                let count = 0
                for(let i=0;i<whoElseIdsDomainWise[domainWhoelse].length; i+=10){
                    queue.create('process-update-folderId-outlook',{whoElses: whoElseIdsDomainWise[domainWhoelse].slice(i,Math.min(i+10,whoElseIdsDomainWise[domainWhoelse].length)), 
                        domainId: domainWhoelse, companyId: req.user.companyId, 
                        emailReported,
                        mailAction:'Deleted' })
                    .removeOnComplete(true).priority('high').save();
                }
            }
            
        }
        if(gsuiteIds.length>0){
            for(let i=0;i<gsuiteIds.length;i++){
                let whoElse = await db.WhoElse.findOne({_id: gsuiteIds[i]})
                if(whoElse.messageId && whoElse.threadId){
                    let { fileName } = await credentials.getCredentials(companyId, whoElse.domainId)
                    await moveEmailGsuite.processMoveEmailDelete({
                        fileName, 
                        email: whoElse.to,
                        threadId: whoElse.threadId,
                        messageId: whoElse.messageId, 
                    })
                    .then(async () => {
                        whoElse.isErrorWhileMoving = {
                            isError: false,
                            message: ''
                        }
                        whoElse.isDeleted = true;
                        await whoElse.save();
                        let isReporter = emailReported?.reporters.find(reporter => reporter.email.toLowerCase() === whoElse.to.toLowerCase())

                        if(emailReported.to !== whoElse.to && !isReporter) mailer.sendResultMailToOtherReporters(whoElse.name ,whoElse.to, whoElse.from, whoElse.subject, emailReported.mailRecievedTime, 'Deleted', companyId)
                        else mailer.sendActionResultMail(whoElse.name ,whoElse.to, whoElse.from, whoElse.subject, emailReported.mailRecievedTime, 'DELETED', false, null, companyId)
                           
                        return ;
                    })
                    .catch(async err => {
                        whoElse.isErrorWhileMoving = {
                            isError: true,
                            message: err
                        }
                        await whoElse.save();
                        return ;
                    })
                } else {
                    whoElse.isErrorWhileMoving = {
                        isError: true,
                        message: 'messageId or threadId is missing'
                    }
                    await whoElse.save();
                }
            }
            }
                
        
     
            updateReportedMailCache(reportedMailId,{isTrashedEmail:"false", isRecoveredEmail:"false", isDeletedEmail:"true"})
            emailReported.isRecoveredEmail = false;
            emailReported.isTrashedEmail = false;
            emailReported.isDeletedEmail = true;
            await emailReported.save()
        return res.status(200).send({ success: true, Message: "Successfully Deleted!"})
    }
    catch (error) {
        console.log("error", error);
        next(error);
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize() , jwtStrategy, checkCompanyValidityStrategy, restrictFeature("TPIR-ARTD"), permission('Emails','Delete'), validateBody(validator), controller);
export default apiRouter;