import Joi from 'joi';
import express from 'express';

import credentials from '../../service/credentials';
import { sanitize } from '../../middlewares/sanitizer';
import { jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import moveEmailGsuite from '../../service/move-email-gsuite';
import moveEmailOutlook from '../../service/move-email-outlook';
import mailer from '../../mailer';
import { permission } from '../../middlewares/permission';
import { queue } from '../../kue'
import { updateReportedMailCache } from '../../functions';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import updateMessageId from '../../service/update-messageId';

let validator = Joi.object().keys({
    whoElseIds: Joi.array().items(
        Joi.string().hex().length(24)                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       
    ).optional(),
    reportedMailId: Joi.string().hex().length(24).required()
});

/**
 * @api {post} /api/v1/mail-actions/move-to-trash Trash Mail
 * @apiName move-to-trash
 * @apiGroup Who-Else
 * @apiDescription Trash a particular who-else mail
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody   {String}  whoElseIds   Required who-else id of the mail to be trashed
 * 
 * @apiSuccess {Boolean} Success    true
 * @apiSuccess {String}  Message    "Successfully trashed and will be update in sometime"
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *       {
 *           "success": true,
 *           "Message": "Successfully deleted and will be update in sometime"
 *       }
 *  
 *  @apiError (500)  {Boolean}  success    false
 *  @apiError (500)  {Array}    messages    ["\"whoElseIds[0]\" is not allowed to be empty"]
 * 
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 500 Not Found
 *      {
 *          "success": false,
 *          "messages": [
 *              "\"whoElseIds[0]\" is not allowed to be empty"
 *          ]
 *      }
 */

let controller = async (req, res, next) => {
    try {
        const { whoElseIds = [], reportedMailId } = req.body;
        const companyId = req.user.companyId;
        const emailReported = await db.ReportedMail.findOne({
            _id: reportedMailId,
            companyId
        })

        if(!emailReported){
            throw new RequestError('invalid reportedMailId recieved');
        }
        if(emailReported.isWhoElseProcessing=='true') throw new RequestError('Who-Else Search in Progress. Please wait and refresh before retrying.'); 
        
        const domain = await db.Domain.findOne({
            _id: emailReported.domainId, deletedAt:null
        })
        if(!domain){
            throw new RequestError('invalid domain or deleted');
        }

        if(emailReported.mailType !== 'Exchange'){
            if(domain.isCredential == false){
            throw new RequestError('Domain credentials not set');   
        }
        }
        
        if(emailReported.mailType == 'Exchange'){
            const user = await db.User.findOne({ email: emailReported.to })
            mailer.sendActionResultMail(user.name ,emailReported.to, emailReported.from, emailReported.subject, emailReported.mailRecievedTime, "SPAM", false, "", companyId)      
            if(emailReported.reporters.length!=0 ){
                emailReported.reporters.forEach(async (email)=>{
                    const user = await db.User.findOne({ email })
                    mailer.sendActionResultMail(user.name ,email, emailReported.from, emailReported.subject, emailReported.mailRecievedTime,"SPAM", false, "", companyId) 
                })
            }
            updateReportedMailCache(reportedMailId,{isTrashedEmail:"true", isRecoveredEmail:"false"})

            emailReported.isRecoveredEmail = false;
            emailReported.isTrashedEmail = true;
            await emailReported.save()
            return res.status(200).send({ success: true, Message: "Successfully marked as Spam!"})
        }
        else{
            let whoElseBatches = [];

            var { token } = await credentials.getCredentials(companyId, domain._id)
            let update = await updateMessageId.processUpdateMessageId(whoElseIds, token)

            for (let i = 0; i < whoElseIds.length; i++) {
                let whoElse = await db.WhoElse.findById(whoElseIds[i]);
                if (!whoElse) continue;
                if(domain.domainType == 'Outlook'){
                    whoElseBatches.push({
                        id: whoElse._id.toHexString(),
                        email: whoElse.to,
                        token: token.access_token,
                        messageId: whoElse.messageId
                    })
                    if(whoElseBatches.length == 20){
                        await moveEmailOutlook.processMoveEmailTrash(whoElseBatches)
                        .then(async (responses) => {
                            queue.create('process-who-else-trash-batches-outlook', {responses, emailReported, Trash:true, Recover:false, Delete:false, companyId})
                            .removeOnComplete(true).priority('high').save();
                            return ;
                        })
                        .catch(async err => {
                            console.log(err)
                            return ;
                        })
                        whoElseBatches = []
                    }
                } else {
                    if(whoElse.messageId && whoElse.threadId){
                        let { fileName } = await credentials.getCredentials(companyId, domain._id)
                        await moveEmailGsuite.processMoveEmailTrash({
                            fileName, 
                            email: whoElse.to,
                            threadId: whoElse.threadId,
                            messageId: whoElse.messageId,
                        })
                        .then(async () => {
                            whoElse.isErrorWhileMoving = {
                                isError: false,
                                message: ''
                            }
                            whoElse.isTrashed = true;
                            await whoElse.save();

                            if(emailReported.to !== whoElse.to) mailer.sendResultMailToOtherReporters(whoElse.name ,whoElse.to, whoElse.from, whoElse.subject, emailReported.mailRecievedTime, 'Trashed', companyId)
                            else mailer.sendActionResultMail(whoElse.name ,whoElse.to, whoElse.from, whoElse.subject, emailReported.mailRecievedTime, 'TRASHED', false, null, companyId)
                           
                            return ;
                        })
                        .catch(async err => {
                            whoElse.isErrorWhileMoving = {
                                isError: true,
                                message: err
                            }
                            await whoElse.save();
                            return ;
                        })
                    } else {
                        whoElse.isErrorWhileMoving = {
                            isError: true,
                            message: 'messageId or threadId is missing'
                        }
                        await whoElse.save();
                    }
                }    
            }
            if(whoElseBatches.length){
                await moveEmailOutlook.processMoveEmailTrash(whoElseBatches)
                        .then(async (responses) => {
                            queue.create('process-who-else-trash-batches-outlook', {responses, emailReported, Trash:true, Recover:false, Delete:false, companyId})
                            .removeOnComplete(true).priority('high').save();
                            return ;
                        })
                        .catch(async err => {
                            console.log(err)
                            return ;
                        })
                        whoElseBatches = [];
            }
            updateReportedMailCache(reportedMailId,{isTrashedEmail:"true", isRecoveredEmail:"false"})
            emailReported.isRecoveredEmail = false;
            emailReported.isTrashedEmail = true;
            await emailReported.save()
            return res.status(200).send({ success: true, Message: "Successfully trashed!"})
        }
        }

    catch (error) {
        console.log("error", error);
        next(error);
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize() , jwtStrategy, permission('Emails','Delete'), validateBody(validator), controller);
export default apiRouter;