import Joi from 'joi';
import express from 'express';

import credentials from '../../service/credentials';
import { sanitize } from '../../middlewares/sanitizer';
import { jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import moveEmailGsuite from '../../service/move-email-gsuite';
import moveEmailOutlook from '../../service/move-email-outlook';
import mailer from '../../mailer';
import { permission } from '../../middlewares/permission';
import { queue } from '../../kue'
import { updateReportedMailCache } from '../../functions';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import updateMessageId from '../../service/update-messageId';

let validator = Joi.object().keys({
    whoElseIds: Joi.array().items(
        Joi.string().hex().length(24)
    ),
    reportedMailId: Joi.string().hex().length(24).required()
});
/**
 * @api {post} /api/v1/mail-actions/recover Recover Mail
 * @apiName recover
 * @apiGroup Who-Else
 * @apiDescription Recover a trashed mail
 * 
 * @apiVersion 1.0.0
 * 
 * @apiQuery   {String}  reportedMailId   Required id of the reported mail
 * 
 * @apiSuccess {Boolean} Success    true
 * @apiSuccess {String}  Message    "Successfully recoverd"
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *       {
 *           "success": true,
 *           "Message": "Successfully recovered"
 *       }
 *  
 *  @apiError (402)  {Boolean}  success-1    false
 *  @apiError (402)  {Array}    messages-1   ["invalid reportedMailId recieved"]
 *  
 *  @apiError (402)  {Boolean}  success-2    false
 *  @apiError (402)  {Array}    results-2    ["Invalid domain credentials"]
 * 
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 402 Not Found
 *      {
 *          "success": false,
 *          "messages": [
 *              "invalid reportedMailId recieved"
 *          ]
 *      }
 * 
 *   @apiErrorExample Error-Response:
 *     HTTP/1.1 500 Not Found
 *      {
 *          "success": false,
 *          "messages": [
 *              "invalid domain credentials"
 *          ]
 *      }
 */

let controller = async (req, res, next) => {    
    try {
        const { whoElseIds = [], reportedMailId } = req.body;
        const companyId = req.user.companyId;
        const emailReported = await db.ReportedMail.findOne({
            _id: reportedMailId,
            companyId
        })
        if(!emailReported){
            throw new RequestError('invalid reportedMailId recieved');
        }
        
        const domain = await db.Domain.findOne({
            _id: emailReported.domainId, deletedAt:null
        })
        if(!domain){
            throw new RequestError('invalid domain or deleted');
        }
        if(emailReported.mailType !== 'Exchange'){
            if(domain.isCredential == false){
            throw new RequestError('Domain credentials not set');   
        }
        }

        if(emailReported.mailType == 'Exchange'){
            const user = await db.User.findOne({ email: emailReported.to })
            if(emailReported.fileName == null) mailer.sendActionResultMail(user.name, emailReported.to, emailReported.from, emailReported.subject, emailReported.mailRecievedTime, "SAFE", false, "", companyId)      
            else mailer.sendActionResultMail(user.name, emailReported.to, emailReported.from, emailReported.subject, emailReported.mailRecievedTime,"SAFE", true, emailReported.fileName, companyId) 
            if(emailReported.reporters.length!=0 ){
                emailReported.reporters.forEach(async (email)=>{
                    const user = await db.User.findOne({ email })
                    if(emailReported.fileName == null) mailer.sendActionResultMail(user.name, email, emailReported.from, emailReported.subject, emailReported.mailRecievedTime, "SAFE", false, "", companyId)      
                    else mailer.sendActionResultMail(user.name, email, emailReported.from, emailReported.subject, emailReported.mailRecievedTime,"SAFE", true, emailReported.fileName, companyId) 
                })
            }
            updateReportedMailCache(reportedMailId,{isTrashedEmail:"false", isRecoveredEmail:"true"})

            emailReported.isRecoveredEmail = true;
            emailReported.isTrashedEmail = false;
            await emailReported.save()
            return res.status(200).send({ success: true, Message: "Successfully recovered!"})
        }
        else{
        let whoElseBatches = []

        if(domain.domainType == 'Outlook') {
            var { token } = await credentials.getCredentials(companyId, domain._id)
            let update = await updateMessageId.processUpdateMessageId(whoElseIds, token)
        }
        for (let i = 0; i < whoElseIds.length; i++) {
            let whoElse = await db.WhoElse.findById(whoElseIds[i]);
            if (!whoElse) continue;
            if(domain.domainType == 'Outlook'){
                var folderId = null
                if(emailReported.to === whoElse.to){ 
                    folderId = emailReported.reporterFolderId}
                else{
                    let findFolderIdInReporters = emailReported.reporters.find(reporter => reporter.email.toLowerCase() === whoElse.to.toLowerCase())
                    if(findFolderIdInReporters) {
                        folderId = findFolderIdInReporters.folderId}
                    else {
                        folderId = whoElse.folderId}
                }
                      
                let { token } = await credentials.getCredentials(companyId, domain._id)
                whoElseBatches.push({
                    id: whoElse._id.toHexString(),
                    email: whoElse.to,
                    token: token.access_token,
                    messageId: whoElse.messageId,
                    folderId
                })
                if(whoElseBatches.length == 20){
                    await moveEmailOutlook.processMoveEmailInbox(whoElseBatches)
                    .then(async (responses) => {
                    queue.create('process-who-else-trash-batches-outlook', {responses, emailReported, Trash:false,Recover:true, Delete:false , companyId})
                    .removeOnComplete(true).priority('high').save();
                    return;
                    })
                    .catch(async err => {
                        console.log(err)
                        return ;
                    })
                    whoElseBatches = []
                }
            } else {
                if(whoElse.messageId && whoElse.threadId){
                    let { fileName } = await credentials.getCredentials(companyId, domain._id)
                    await moveEmailGsuite.processMoveEmailInbox({
                        fileName, 
                        email: whoElse.to,
                        threadId: whoElse.threadId,
                        messageId: whoElse.messageId,
                    })
                    .then(async () => {
                        whoElse.isErrorWhileMoving = {
                            isError: false,
                            message: ''
                        }
                        whoElse.isTrashed = false;
                        await whoElse.save();

                        if(emailReported.to !== whoElse.to) mailer.sendResultMailToOtherReporters(whoElse.name ,whoElse.to, whoElse.from, whoElse.subject, emailReported.mailRecievedTime, 'Recovered', companyId)
                        else mailer.sendActionResultMail(whoElse.name ,whoElse.to, whoElse.from, whoElse.subject, emailReported.mailRecievedTime, 'RECOVERED', false, null, companyId)
                        
                        return ;
                    })
                    .catch(async err => {
                        whoElse.isErrorWhileMoving = {
                            isError: true,
                            message: err
                        }
                        await whoElse.save();
                        return ;
                    })
                } else {
                    whoElse.isErrorWhileMoving = {
                        isError: true,
                        message: 'messageId or threadId is missing'
                    }
                    await whoElse.save();
                }
            }    
        }
        if(whoElseBatches.length){
            await moveEmailOutlook.processMoveEmailInbox(whoElseBatches)
                    .then(async (responses) => {
                    queue.create('process-who-else-trash-batches-outlook', {responses, emailReported, Trash:false,Recover:true, Delete:false, companyId })
                    .removeOnComplete(true).priority('high').save();
                    return ;
                    })
                    .catch(async err => {
                        console.log(err)
                        return ;
                    })
                    whoElseBatches = [];
        }
        updateReportedMailCache(reportedMailId,{isTrashedEmail:"false", isRecoveredEmail:"true"})
        emailReported.isRecoveredEmail = true;
        emailReported.isTrashedEmail = false;
        await emailReported.save()
        return res.status(200).send({ success: true, Message: "Successfully recovered!"})
    }
    }
    catch (error) {
        console.log("error", error);
        next(error);
    }
}

const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), jwtStrategy, permission('Emails','Write'), validateBody(validator), controller);
export default apiRouter;