import Joi from 'joi';
import express from 'express';

import { sanitize } from '../../middlewares/sanitizer';
import { checkCompanyValidityStrategy, jwtStrategy } from '../../middlewares/strategy';
import { permission } from '../../middlewares/permission';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';

let validator = Joi.object().keys({
    employeeId: Joi.string().hex().length(24).required(),
    policyData: Joi.array().items({
        name: Joi.string().required(),
        value: Joi.array().items(Joi.any().valid('Read', 'Write', 'Delete')).required(),
    }).required()
});

/**
 * @api {post} /api/v1/policy/add Add Policy
 * @apiName add-policy
 * @apiGroup Policy
 * @apiDescription Add Policies to an IAM user
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody   {String}  employeeId   Required Id of the IAM emplyoee
 * @apiBody   {Array}  policyData    Required key value pair of name of the policy and value of that particular policy ('Read', 'Write', 'Delete').
 * 
 * 
 * @apiSuccess {Boolean} Success    true
 * @apiSuccess {String}  Message    "Policy updated successfully"
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *       {
 *           "success": true,
 *           "Message": "Policy updated successfully"
 *       }
 *  
 *  @apiError (404)  {Boolean}  success     false
 *  @apiError (404)  {Array}    messages    Employee not Found
 * 
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 404 Not Found
 *      {
 *          "success": false,
 *          "messages": [
 *              "Employee not Found"
 *          ]
 *      }
 */

let controller = async (req, res, next) => {
    try {
        const { policyData, employeeId } = req.body;
        const companyId = req.user.companyId;
        const employee = await db.Employee.findOne({
            _id: employeeId,
            companyId
        });
        if (!employee) {
            throw new RequestError('Employee not Found', 404);
        }
        for(let i=0; i<policyData.length; i++){
            let { name, value } = policyData[i]
            let policy = await db.EmployeePolicy.findOne({
                employeeId, name
            })
            if(policy){
                await db.EmployeePolicy.updateOne({
                    _id: policy._id
                }, {
                    value
                })
            } else {
                await db.EmployeePolicy.create({
                    employeeId, name, value
                })
            }
        }
        res.status(201).send({ success: true, message: "Policy updated successfully"})
    } catch (err) {
        console.log(err);
        next(err)
    }
}

const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), validateBody(validator), jwtStrategy, checkCompanyValidityStrategy, permission('IAM','Write'), controller);
export default apiRouter;