import express from 'express';

import config from '../../config';
import { jwtStrategy } from '../../middlewares/strategy';

/**
 * @api {get} /api/v1/profile/details details
 * @apiName details
 * @apiGroup Profile
 * @apiDescription Details of the user 
 * 
 * @apiVersion 1.0.0
 * 
 * @apiSuccess {Boolean} success              true
 * @apiSuccess {Object}  user[security]       object of security configurations of the user
 * @apiSuccess {id}      _id                  id of the user
 * @apiSuccess {String}  name                 name of the user
 * @apiSuccess {Object}  company              details of the company of user
 * @apiSuccess {String}  url                  url of the logo
 * @apiSuccess {String}  isTsat               url of TSAT
 * @apiSuccess {String}  isTsat               status of sending the mail to tsat
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *      {
 *          "success": true,
 *          "user": {
 *              "security": {
 *                  "status": false,
 *                  "otpType": "MAIL",
 *                  "otp": null,
 *                  "tOtpKey": null,
 *                  "otpExpiry": null
 *              },
 *              "_id": "63d0ea61032f943fc31e4e82",
 *              "name": "sarvesh patel",
 *              "email": "sarvesh.patel@kratikal.com"
 *          },
 *          "company": {
 *              "_id": "63d0ea61032f943fc31e4e80",
 *              "name": "Kratikal Tech Pvt Limited",
 *              "logo": null,
 *              "socEmail": "sarvesh.patel@kratikal.com"
 *          },
 *          "url": "https://new-stag.threatalertbutton.com/static/logo/"
 *          "tsatUrl": "https://tsat.threatcop.com"
 *          "isTsat": false
 *      }   
 * 
 */

let controller = async (req, res, next) => {
    try {
        const userId = req.user._id;
        const companyId = req.user.companyId;
        let user = null;
        let smtp = {
            status: false,
            host: null,
            port: null,
            password: null,
            username: null,
        }
        if(req.user.type == 'root'){
            user = await db.Admin.findOne({
                _id: userId
            }, 'firstName lastName email security')
        } else {
            user = await db.Employee.findOne({
                _id: userId
            }, 'firstName lastName email security')
        }
        const company = await db.Company.findById(companyId, 'name logo socEmail licenceKey broadcastThreshold reportingMethod');
        
        let tsatUrl = await db.MetaData.findOne({
            companyId,
            name: 'tsatUrl'
        })
        let isTsat = await db.MetaData.findOne({
            companyId,
            name: 'isTsat'
        })

        if(!tsatUrl){
            tsatUrl = await db.MetaData.create({
                companyId,
                name:'tsatUrl',
                value: `${config.app.tsatUrl}`
            })
            isTsat = await db.MetaData.create({
                companyId,
                name:'isTsat',
                value:"false"
            })
        }

        let userLimitMessenger = await db.MetaData.findOne({
            companyId,
            name: 'userLimitMessenger'
        })

        if(!userLimitMessenger){
            userLimitMessenger = await db.MetaData.create({
                companyId,
                name:'userLimitMessenger',
                value:'0'
            })
        }

        let isSmtp = await db.MetaData.findOne({
            companyId,
            name:'mailSmtp'
        })
        if(!isSmtp){ 
            smtp.status = false;
            await db.MetaData.findOneAndUpdate({name: "mailSmtp", companyId },{ value: false},{ upsert: true, new: true, setDefaultsOnInsert: true })      
        }
        else if(isSmtp.value == 'false') smtp.status = false;
        else{
            smtp.status = isSmtp.value
            smtp.host =(await db.MetaData.findOne({name:"smtpHost",companyId:companyId})).value
            smtp.port = (await db.MetaData.findOne({name:"smtpPort",companyId:companyId})).value
            smtp.username = (await db.MetaData.findOne({name:"smtpUsername",companyId:companyId})).value
            smtp.from = (await db.MetaData.findOne({name:"smtpFrom",companyId:companyId})).value
        }
        

        let timeZone = await db.MetaData.findOne({name:"timeZone", companyId})
        if(!timeZone) timeZone = 'Asia/Kolkata'
        else timeZone = timeZone.value

        return res.status(200).send({ 
            success: true, 
            user: {
                firstName: user.firstName,
                lastName: user.lastName,
                email: user.email,
                is2fa: user.security.status,
                otpType: user.security.otpType
            }, 
            company,
            url: `${config.app.url}/api/static/logo/`,
            tsatUrl: tsatUrl.value,
            isTsat: isTsat.value,
            timeZone,
            smtp,
        })
    } catch (error) {
        console.log(error);
        next(error);
    }
}

const apiRouter = express.Router();
apiRouter.route('/').get(jwtStrategy, controller);
export default apiRouter;

