import Joi from 'joi';
import path from 'path';
import multer from 'multer';
import express from 'express';

import { jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { permission } from '../../middlewares/permission';
import { scanFile } from '../../middlewares/scanAndUploadFile';
import sanitizeHtml from "sanitize-html";
import config from '../../config';


let profilePicturePath = path.join(path.dirname(require.main.filename), 'public', 'logo');

let storage = multer.diskStorage({
    destination: function (req, file, cb) {
        cb(null, profilePicturePath);
    },
    filename: function (req, file, cb) {
        cb(null, req.user.companyId + path.extname(file.originalname));
    }
})
let uploadAttachment = multer({ storage: storage, limits: { fileSize: 10485760 } })

const validator = Joi.object().keys({
    firstName: Joi.string().required(),
    lastName: Joi.string().required(),
    companyName: Joi.string().required(),
	logo: Joi.string().optional().allow(null).empty('')
});

/**
 * @api {post} /api/v1/profile/update-profile update-profile
 * @apiName update-profile
 * @apiGroup Profile
 * @apiDescription Update profile details of the user 
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody    {String}  name          Name of the user
 * @apiBody    {String}  companyName   Company name of the user
 * @apiBody    {String}  [logo]        Logo of the user
 *
 * @apiSuccess {String}   success      true
 * @apiSuccess {String}   message      Details updated successfully
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *      {
 *          "success": true,
 *          "user":    "Details updated successfully"
 *      } 
 * 
 *  @apiError (400)  {Boolean} success    false
 *  @apiError (400)  {Array}   messages   "Only PNG or JPG files are accepted
 * 
 *  @apiErrorExample Error-Response:
 *     HTTP/1.1 400 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *              "Only PNG or JPG files are accepted"
 *           ]
 *       }  
 * 
 */

let controller = async (req, res, next) => {
    try {
        let { firstName, lastName, companyName } = req.body
        
        firstName = sanitizeHtml(firstName, { allowedTags: [], allowedAttributes: {}})
        lastName = sanitizeHtml(lastName, { allowedTags: [], allowedAttributes: {}})
        companyName = sanitizeHtml(companyName, { allowedTags: [], allowedAttributes: {}})

        const image = req.file ? req.file.filename : null;
        if (req.file && (req.file.mimetype != 'image/png' && req.file.mimetype != 'image/jpeg' && req.file.mimetype != 'image/jpg') || req?.file?.size >= 5000000) {
            throw new RequestError('Only PNG or JPG files are accepted with size less than 5mb!',406);
        }

        let company = await db.Company.findById(req.user.companyId)
        if(!company) throw new RequestError('Company not found!',404)
        
        if(req.user.type == 'root'){
            await db.Admin.findOneAndUpdate({ 
                _id: req.user._id 
            }, { 
                firstName, lastName 
            });
        }
        else{
            await db.Employee.findOneAndUpdate({ 
                _id: req.user._id 
            }, { 
                firstName, lastName 
            });
        }

        let data = {
            name: companyName
        }
        if(image) data['logo'] = image;

        const updatedCompany = await db.Company.findOneAndUpdate(
            { _id: req.user.companyId },  
            { $set: data },               
            { new: true }                 
        );

        return res.status(200).send({ success: true, message: 'Details updated successfully', dataUrl: `${config.app.url}/api/static/logo/${updatedCompany.logo}` })

    } catch (err) {
        console.log(err);
        next(err);
    }
}

const apiRouter = express.Router();
apiRouter.route('/').post(jwtStrategy, permission('Settings','Write'), uploadAttachment.single('logo'), scanFile, validateBody(validator), controller);
export default apiRouter;