import Joi from 'joi';
import Spamc from 'spamc';
import express from 'express';

import config from '../../config';
import { queue } from '../../kue';
import parseHeader from '../../service/parse-header';
import { validateBody } from '../../middlewares/validator';

const spamc = new Spamc(config.spamassession.host, config.spamassession.port, 10000);

const validator = Joi.object().keys({
    mailBody: Joi.string().required(),
    threadId: Joi.string().required(),
    messageId: Joi.string().required(),
    email: Joi.string().email().required(),
});

/**
 * @api {post} /api/report/check-email check-email
 * @apiName Check Email
 * @apiGroup Report
 * @apiDescription check the email of the user
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {String} mailBody               Required mail data.
 *
 * @apiSuccess {Boolean} success             true
 * @apiSuccess {String}  headerDetails       header details of the message.
 * @apiSuccess {String}  result              spam result of the message.
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 404 Not Found
 *      {
 *          "success":          true,
 *          "result":           "Details updated successfully",
 *          "headerDetails"     []
 *      } 
 * 
 */


const controller = async (req, res, next) => {
    try {
        const { mailBody, messageId, threadId, email } = req.body;
        console.time(`process-header-${messageId}`);
        let headerDetails = await parseHeader.processHeader(mailBody);
        if(Object.keys(headerDetails).length == 0){
            console.timeEnd(`process-header-${messageId}`);
            throw new RequestError('Error parsing header details', 500);
        }
        console.timeEnd(`process-header-${messageId}`);
        console.time(`spamassession-${messageId}`)
        new Promise((resolve, reject) => {
            spamc.report(mailBody, function (err, result) {
                if(err) reject(err);
                else resolve(result);
            })
        })
        .then(result => {
            console.timeEnd(`spamassession-${messageId}`)
            queue.create('check-email-spam-or-not-event-gsuite', {
                email,
                mailBody, 
                threadId,
                messageId,
                headerDetails, 
                mailType: 'Gsuite',
                score: result && result.spamScore? result.spamScore: 0
            })
            .save().removeOnComplete(true).priority('high');
            result.spamScore = result?.spamScore? result.spamScore: 0
            res.status(200).json({ success: true, result, headerDetails })
        })
        .catch(err => {
            console.log(err);
            next(err);
        })

    } catch(err){
        console.log(err);
        next(err);
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post(validateBody(validator), controller);
export default apiRouter;