import Joi from 'joi';
import Spamc from 'spamc';
import express from 'express';

import config from '../../config';
import { queue } from '../../kue';
import parseHeader from '../../service/parse-header';
import getMailData from '../../service/get-mail-data';
import { sanitize } from '../../middlewares/sanitizer';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';

const spamc = new Spamc(config.spamassession.host, config.spamassession.port, 10000);

const validator = Joi.object().keys({
    messageId: Joi.string().required(),
    accessToken: Joi.string().required(),
    email: Joi.string().email().required(),
});

/**
 * @api {post} /api/report/check-email check-email
 * @apiName Check Email
 * @apiGroup Report
 * @apiDescription check the email of the user
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {String} messageId               Required message id of the mail.
 * @apiBody {String} accessToken             Required access token of the mailbox.
 *
 * @apiSuccess {Boolean} success             true
 * @apiSuccess {String}  headerDetails       header details of the message.
 * @apiSuccess {String}  result              spam result of the message.
 * 
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 404 Not Found
 *      {
 *          "success":          true,
 *          "result":           "Details updated successfully",
 *          "headerDetails"     []
 *      } 
 * 
 */

const controller = async (req, res, next) => {
    try {
        const { accessToken, messageId, email } = req.body;
        console.time('process mail data')
        let header = await getMailData.processHeader(messageId, accessToken);
        console.timeEnd('process mail data')
        console.time('process header');
        let headerDetails = await parseHeader.processHeader(header);
        console.timeEnd('process header');
        console.time('spamassession')
        new Promise((resolve, reject) => {
            spamc.headers(header, function (err, result) {
                if(err) reject(err);
                else resolve(result);
            })
        })
        .then(result => {
            console.timeEnd('spamassession')
            queue.create('check-email-spam-or-not-event-outlook', {
                email,
                header,
                messageId, 
                headerDetails, 
                mailType: 'Outlook',
                score: result && result.spamScore? result.spamScore: 0
            })
            .save().removeOnComplete(true).priority('high');
            res.status(200).json({ success: true, result, headerDetails })
        })
        .catch(err => {
            console.log(err);
            next(err);
        })

    } catch(err){
        console.log(err);
        next(err);
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), validateBody(validator), controller);
export default apiRouter;