import Joi from 'joi';
import express from 'express';

import { queue } from '../../kue';
import parseEmail from '../../service/parse-email';
import Spamc from 'spamc';
import { validateBody } from '../../middlewares/validator';
import mailer from '../../mailer';
import config from '../../config';
import getUrls from 'get-urls';
import decodeUriComponent from 'decode-uri-component';
import canAccessFeature from '../../service/allowed-features';



const spamc = new Spamc(config.spamassession.host, config.spamassession.port, 10000);


let validator = Joi.object().keys({
    mailBody: Joi.string().required(),
    threadId: Joi.string().required(),
    messageId: Joi.string().required(),
    email: Joi.string().email().required(),
    internetMessageId: Joi.string().required(),
});

const checkUrlsForRid = async (urls) => {
    try {
        let urlContainsRid = false
        const urlsArray = Array.from(urls);

        for(let i=0; i<urlsArray.length;i++){
            let url = urlsArray[i]
            let decodedUrl = decodeURIComponent(url);
            if(decodedUrl.indexOf('rid=')>-1){
                urlContainsRid = decodedUrl
                break;
            }
        }
        return urlContainsRid
        
    } catch (err) {
        console.log(err);
        return false;
    }
}

// const spamc = new Spamc(config.spamassession.host, config.spamassession.port, 10000);


/**
 * @api {post} /api/report/report-email report-email
 * @apiName Report Mail
 * @apiGroup Report
 * @apiDescription Report a mail
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {String} accessToken       Required accessToken of the mail.
 * @apiBody {String} messageId         Required messageId of the mail.
 * @apiBody {String} email             Required email of the mail.
 *
 * @apiSuccess {Boolean} success    true
 * @apiSuccess {String} message    'email reported successfully'
 * 
  * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *      {
 *          "success":  true,
 *          "message":  'email reported successfully'        
 *      } 
 * 
 * @apiError {Boolean}    success     false
 * @apiError {Boolean}    messages    ["user not exists"]
 * 
 * 
 * @apiErrorExample Error-Response:
 *     HTTP/1.1 404 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *               "user not exists"
 *           ]
 *       }
 */

const moveToTrash = (companyId, email, domainId, internetMessageId) =>{
    return new Promise (async (resolve,reject)=>{
        const isWhoElse = await db.WhoElse.findOne({ internetMessageId, to: email, companyId, domainId})
        if(isWhoElse){
            isWhoElse.isTrashed = true;
            await isWhoElse.save()
            resolve(true)
        }
        else resolve(false);
    })
}

let controller = async (req, res, next) => {
    try {
        let { mailBody, email, messageId, threadId, internetMessageId } = req.body;
        console.time(`find-user-${messageId}`)
        const user = await db.User.findOne({ email: email.toLowerCase(), isActive: true, emailType: "Gsuite",  deletedAt: null })
        console.timeEnd(`find-user-${messageId}`)
        if(!user){
            throw new RequestError('user not exists', 404);
        }

        const companyRemainingTime = await db.MetaData.findOne({name:'expiredAt', companyId: user.companyId })
        if(new Date() > new Date(companyRemainingTime.value)) throw new RequestError('License expired', 404)

        const userId = user._id;
        const domainId = user.domainId;
        const companyId = user.companyId;

        const domain = db.Domain.findOne({
            _id: domainId,
            companyId,
            deletedAt: null
        })
        if(!domain){
            throw new RequestError('domain not exists', 404);
        }


        const isReported = await db.ReportedMail.findOne({ internetMessageId })
        if(isReported){
            if(isReported.isRecoveredEmail === 'false'){
            let moveTrash = await moveToTrash(companyId, email, domainId, internetMessageId)
            if(!moveTrash) console.log('Email Not Found in Who-Else for the reported mail!')
            if(await canAccessFeature(companyId,"TPIR-RN")) mailer.sendAlreadyReportedMail(user.name, email, companyId)
            isReported.reporters.push({email})
            await isReported.save()
            res.status(200).json({ success: true, message: 'email reported successfully', reporters: isReported.reporters ? isReported.reporters.length + 1 : 0, spamScore: isReported.spamScore});
            }
            else return res.status(200).json({ success: true, message: 'Email already reported by someone and has been found safe!', moveToJunk: false });
        }
        else{
        console.time(`process-mail-body-${messageId}`)
        const parsedEML = await parseEmail.parseEml(mailBody);
        let body = parsedEML.html;
        let header = parsedEML
        let plainHeader = mailBody.split('Content-Type: text/plain;')[0];
        let attachments = parsedEML?.attachments??[];
        console.timeEnd(`process-mail-body-${messageId}`)
        
        const tsatUrl = await db.MetaData.findOne({
            companyId,
            name: 'tsatUrl'
        })
        const isTsat = await db.MetaData.findOne({
            companyId,
            name: 'isTsat'
        })
                
        let isTsatMail = isTsat.value === 'true';

        let urlsInbody = await getUrls(body)
        let isRidInBody = false
        if(urlsInbody.size > 0 ) isRidInBody = await checkUrlsForRid(urlsInbody)
            
        if (await canAccessFeature(companyId,"TPIR-TSAT") && isRidInBody != false && isTsatMail && tsatUrl.value) {
            // call to tsat for email reporting
            queue.create('route-email-tsat', { header, body: decodeUriComponent(`${body}`), tsatUrl })
                .removeOnComplete(true).priority('high').save();
        }

        new Promise(async (resolve, reject) => {
            console.time('spamassession'+messageId)
            if(await canAccessFeature(companyId,"TPIR-SSRE")){
                spamc.headers(plainHeader, function (err, result) {
                    if(err) reject(err);
                    else resolve(result);
                })
            }
            else resolve(0)
        }).then(result => {
            console.log("SpamScore---->",result)
            console.timeEnd('spamassession'+messageId)
        queue.create('reported-email-gsuite', {
            email, 
            header,
            plainHeader, 
            body, 
            userId, 
            userName: user ? user.name ? user.name :'' :'',
            threadId, 
            companyId,
            messageId, 
            internetMessageId,
            attachments, 
            domainId,
            spamScore:result && result.spamScore? result.spamScore: 0,
            isCheckHeader: true,
            isTsatMail: isRidInBody != false,
            mailBody
        }).removeOnComplete(true).priority('high').save();

        res.status(200).json({ success: true, message: 'email reported successfully',reporters: 1, moveToJunk: true, spamScore: result && result.spamScore? result.spamScore: 0 });
        })

    }
    } catch(err){
        console.log('From Report-email-gsuite',err);
        next(err);
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post(validateBody(validator), controller);
export default apiRouter;
