import Joi from 'joi';
import express from 'express';

import { queue } from '../../kue';
import getMailData from '../../service/get-mail-data';
import Spamc from 'spamc';
import config from '../../config';
import { sanitize } from '../../middlewares/sanitizer';
import { validateBody } from '../../middlewares/validator';
import parseHeader from '../../service/parse-header';
import parseEmail from '../../service/parse-email';
import mailer from '../../mailer'
import canAccessFeature from '../../service/allowed-features';
import getIp from '../../service/get-ip';

// let validator = Joi.object().keys({
//     header: Joi.string().required(),
//     body: Joi.string().required(),
//     email: Joi.string().email().required(),
//     internetMessageId: Joi.string().required(),
// });

const spamc = new Spamc(config.spamassession.host, config.spamassession.port, 10000);
/**
 * @api {post} /api/report/report-email report-email
 * @apiName Report Mail
 * @apiGroup Report
 * @apiDescription Report a mail
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {String} accessToken       Required accessToken of the mail.
 * @apiBody {String} messageId         Required messageId of the mail.
 * @apiBody {String} email             Required email of the mail.
 *
 * @apiSuccess {Boolean} success    true
 * @apiSuccess {String} message    'email reported successfully'
 * 
  * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *      {
 *          "success":  true,
 *          "message":  'email reported successfully'        
 *      } 
 * 
 * @apiError {Boolean}    success     false
 * @apiError {Boolean}    messages    ["user not exists"]
 * 
 * 
 * @apiErrorExample Error-Response:
 *     HTTP/1.1 404 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *               "user not exists"
 *           ]
 *       }
 */

const getParseHeader = async (header, status = true) => {
    try {
        let investigate = {};
        let parseEml = header
        if (status) parseEml = await parseEmail.parseHeader(header);
        if (parseEml) {
            let from = (parseEml && parseEml.from && parseEml.from[0].address) ? parseEml.from[0].address : '';
            let spfDomain = parseEml.headers['return-path'] ?
                Array.isArray(parseEml.headers['return-path']) ?
                    parseEml.headers['return-path'][0].split("@")[1] :
                    parseEml.headers['return-path'].split("@")[1] :
                null;
            let parsedDomain = from.split('@')[1];
            let fromDomain = parsedDomain;
            investigate['spfAllignment'] = (fromDomain == spfDomain);
            let dkimSiganture = parseEml.headers['dkim-signature']
            if (Array.isArray(dkimSiganture)) {
                let search = `d=${fromDomain}`
                let filteredDkimSiganture = dkimSiganture.filter(r => r.includes(search))
                if (filteredDkimSiganture.length > 0) dkimSiganture = filteredDkimSiganture[0];
                else if (dkimSiganture > 0) dkimSiganture = dkimSiganture[0];
                else dkimSiganture = null;
            }
            let authResults = parseEml.headers['authentication-results'];
            authResults = ((Array.isArray(authResults) && authResults.length > 0) ? authResults.filter(r => r.startsWith('amazonses.com')).join(" ") : authResults);
            let dkimDomain = ((dkimSiganture && dkimSiganture != null) ?
                (function () {
                    let dIndex = dkimSiganture.search(/d=.*?;/);
                    let dValue = dkimSiganture.slice(dIndex + 2, dkimSiganture.indexOf(";", dIndex + 2));
                    return dValue;
                })() : null);

            parsedDomain = dkimDomain ? await parseDomain(dkimDomain) : null;
            dkimDomain = parsedDomain ? `${parsedDomain.domain}` : dkimDomain;
            investigate['dkimAllignment'] = (fromDomain == dkimDomain);
            investigate['senderIp'] = await getipAddress(parseEml.headers['received']);
            let { country, continent } = await getIp.processIp(investigate['senderIp']);
            if (country && continent) {
                investigate['country'] = country;
                investigate['continent'] = continent
            } else {
                investigate['country'] = 'NOT FOUND';
                investigate['continent'] = 'NOT FOUND';
            }
            investigate['spf'] = stringSearcher(authResults, 'spf=pass');
            investigate['dkim'] = stringSearcher(authResults, 'dkim=pass');
            investigate['dmarc'] = stringSearcher(authResults, ['dmarc=pass', 'dmarc=none']);
            return investigate;
        } else {
            console.error('error from parsing header details');
            return {};
        }
    } catch (err) {
        console.log(err);
        return {};
    }
}

const getipAddress = function (receivedHeaders) {
    let ip = null;
    receivedHeaders.
        forEach(header => {
            const match = header.match(/\b(?:\d{1,3}\.){3}\d{1,3}\b/);
            if (match) {
                ip = match[0]
            }
        });
    return ip;
}

const stringSearcher = function (str, key) {
    let keyList = [];
    if (typeof key === 'string') keyList.push(key);
    else if (Array.isArray(key)) keyList = key;
    if (str && str.length > 0) {
        if (key.length == 0) return 'fail';
        let filteredList = keyList.filter((iKey) => str.indexOf(iKey) != -1)
        if (filteredList.length > 0) {
            return 'pass';
        } else return 'fail';
    } else {
        return 'fail';
    }
}

let controller = async (req, res, next) => {
    try {
        let { messageId, email, internetMessageId, headers, body, from, subject, cc, toName = '', bcc = [], mailRecievedTime } = req.body;
        const user = await db.User.findOne({ email })
        console.log({from});
        if(!user){
            throw new RequestError('user not exists', 404);
        }
        const userId = user._id;
        const domainId = user.domainId;
        const companyId = user.companyId;

        const domain = await db.Domain.findOne({
            _id: domainId,
            companyId,
            deletedAt: null
        })
        if(!domain){
            throw new RequestError('domain not exists', 404);
        }

        const tsatUrl = await db.MetaData.findOne({
            companyId,
            name: 'tsatUrl'
        })
        const isTsat = await db.MetaData.findOne({
            companyId,
            name: 'isTsat'
        })

                    

        if(await canAccessFeature(companyId,"TPIR-TSAT") && body.indexOf('?rid=')!== -1 && isTsat.value == 'true' && tsatUrl.value){
            // call to tsat for email reporting
            queue.create('route-email-tsat', { header:headers, body, tsatUrl})
            .removeOnComplete(true).priority('high').save();

            if(await canAccessFeature(companyId,"TPIR-RN")){
                queue.create('send-reported-mail', {
                    to: email,
                    userName: user ? user.name ? user.name :'' :'',
                    from: from.emailAddress,
                    subject,
                    mailRecievedTime,
                    companyId
                }).removeOnComplete(true).priority('high').save()
            }

            if(await canAccessFeature(companyId,"TPIR-EASOC")){
            queue.create('send-soc-mail', {
                to:email,
                from: from.emailAddress,
                subject,
                mailRecievedTime,
                header: headers,
                body,
                companyId,
            }).removeOnComplete(true).priority('high').save()
            }

            return res.status(200).json({ success: true, message: 'email reported successfully', reporters: 1, moveToJunk: true, spamScore: 0 });        
        }

        const isReported = await db.ReportedMail.findOne({ internetMessageId })

        if(isReported){
                await db.ReportedMail.updateOne({ _id:isReported._id }, { $push: { reporters: {email} } });
                if(await canAccessFeature(companyId,"TPIR-RN")) mailer.sendAlreadyReportedMail(user.name, email, companyId)
                res.status(200).json({ success: true, reporters: isReported.reporters ? isReported.reporters.length + 2 : 0, spamScore: isReported.spamScore,moveToJunk: true, message: 'email reported successfully'  });
        }
        else{

            if(domain.domainTypeForOutlook == 'EXCHANGE'){
                let exchangeHeaders = ''
                for (let i = 0; i < headers.length; i++) {
                    exchangeHeaders = exchangeHeaders + headers[i]?.name + ' ' + headers[i]?.value + '\n';
                }
                headers = exchangeHeaders
            }


            let headerDetails = {};
            const parseEml = await parseEmail.parseEml(headers)

            if(domain.domainTypeForOutlook == "EXCHANGE") { headerDetails = await getParseHeader(headers)}
            else { headerDetails = await parseHeader.processHeader(parseEml, false)}
            let otherTo = parseEml.to ? parseEml.to : [];
            
            const emailReported = await db.ReportedMail.create({
                from: from.emailAddress, fromName: from.displayName, to: email, toName:user.name, subject, mailId: messageId, cc,
                bcc, otherTo:[], messageId, internetMessageId, mailRecievedTime, companyId, isLiteEnvReport: true,
                mailType: user.emailType , header: headers, body, userId, headerDetails, domainId, parsedHeader: parseEml
            })
            if(await canAccessFeature(companyId,"TPIR-RN")){
            queue.create('send-reported-mail', {
                to: email,
                userName: user ? user.name ? user.name :'' :'',
                from: from.emailAddress,
                subject,
                mailRecievedTime,
                companyId
            }).removeOnComplete(true).priority('high').save()
            }
            if(await canAccessFeature(companyId,"TPIR-EASOC")){
            queue.create('send-soc-mail', {
                to: email,
                from: from.emailAddress,
                subject,
                mailRecievedTime,
                header: headers,
                body,
                companyId,
            }).removeOnComplete(true).priority('high').save()
            }

        res.status(200).json({ success: true, message: 'email reported successfully', reporters: 1, moveToJunk: true, spamScore: 0 });
        
    }
    } catch(err){
        console.log(err);
        next(err);
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post(controller);
export default apiRouter;

