import Joi from 'joi';
import express from 'express';

import { queue } from '../../kue';
import Spamc from 'spamc';
import config from '../../config';
import { sanitize } from '../../middlewares/sanitizer';
import { validateBody } from '../../middlewares/validator';
import parseEmail from '../../service/parse-email'
import parseDomain from "parse-domains";
import getIp from '../../service/get-ip';
import path from 'path'
import fs from 'fs'
import Files from '../../Content/upload_file'
import getUrls from 'get-urls';
import decodeUriComponent from 'decode-uri-component';
import canAccessFeature from '../../service/allowed-features';



let validator = Joi.object().keys({
    messageId: Joi.string().required(),
    accessToken: Joi.string().required(),
    email: Joi.string().email().required(),
    internetMessageId: Joi.string().required(),
    header: Joi.required(),
    body: Joi.string().required(),
    from: Joi.string().required(),
    subject: Joi.string().required(),
    cc: Joi.string().required(),
    toName: Joi.string().required(),
    bcc: Joi.string().required(),
    mailRecievedTime: Joi.date().required(),
    mimeContent: Joi.string().required()
});


const spamc = new Spamc(config.spamassession.host, config.spamassession.port, 10000);
/**
 * @api {post} /api/report/report-email report-email
 * @apiName Report Mail
 * @apiGroup Report
 * @apiDescription Report a mail
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {String} accessToken       Required accessToken of the mail.
 * @apiBody {String} messageId         Required messageId of the mail.
 * @apiBody {String} email             Required email of the mail.
 *
 * @apiSuccess {Boolean} success    true
 * @apiSuccess {String} message    'email reported successfully'
 * 
  * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *      {
 *          "success":  true,
 *          "message":  'email reported successfully'        
 *      } 
 * 
 * @apiError {Boolean}    success     false
 * @apiError {Boolean}    messages    ["user not exists"]
 * 
 * 
 * @apiErrorExample Error-Response:
 *     HTTP/1.1 404 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *               "user not exists"
 *           ]
 *       }
 */

const getipAddress = function (receivedHeaders) {
    let ip = null;
    receivedHeaders.
        forEach(header => {
            const match = header.match(/\b(?:\d{1,3}\.){3}\d{1,3}\b/);
            if (match) {
                ip = match[0]
            }
        });
    return ip;
}

const stringSearcher = function (str, key) {
    let keyList = [];
    if (typeof key === 'string') keyList.push(key);
    else if (Array.isArray(key)) keyList = key;
    if (str && str.length > 0) {
        if (key.length == 0) return 'fail';
        let filteredList = keyList.filter((iKey) => str.indexOf(iKey) != -1)
        if (filteredList.length > 0) {
            return 'pass';
        } else return 'fail';
    } else {
        return 'fail';
    }
}

const getParseHeader = async (header, status = true) => {
    try {
        let investigate = {};
        let parseEml = header
        if (status) parseEml = await parseEmail.parseHeader(header);
        if (parseEml) {
            let from = (parseEml && parseEml.from && parseEml.from[0].address) ? parseEml.from[0].address : '';
            let spfDomain = parseEml.headers['return-path'] ?
                Array.isArray(parseEml.headers['return-path']) ?
                    parseEml.headers['return-path'][0].split("@")[1] :
                    parseEml.headers['return-path'].split("@")[1] :
                null;
            let parsedDomain = from.split('@')[1];
            let fromDomain = parsedDomain;
            investigate['spfAllignment'] = (fromDomain == spfDomain);
            let dkimSiganture = parseEml.headers['dkim-signature']
            if (Array.isArray(dkimSiganture)) {
                let search = `d=${fromDomain}`
                let filteredDkimSiganture = dkimSiganture.filter(r => r.includes(search))
                if (filteredDkimSiganture.length > 0) dkimSiganture = filteredDkimSiganture[0];
                else if (dkimSiganture > 0) dkimSiganture = dkimSiganture[0];
                else dkimSiganture = null;
            }
            let authResults = parseEml.headers['authentication-results'];
            authResults = ((Array.isArray(authResults) && authResults.length > 0) ? authResults.filter(r => r.startsWith('amazonses.com')).join(" ") : authResults);
            let dkimDomain = ((dkimSiganture && dkimSiganture != null) ?
                (function () {
                    let dIndex = dkimSiganture.search(/d=.*?;/);
                    let dValue = dkimSiganture.slice(dIndex + 2, dkimSiganture.indexOf(";", dIndex + 2));
                    return dValue;
                })() : null);

            parsedDomain = dkimDomain ? await parseDomain(dkimDomain) : null;
            dkimDomain = parsedDomain ? `${parsedDomain.domain}` : dkimDomain;
            investigate['dkimAllignment'] = (fromDomain == dkimDomain);
            investigate['senderIp'] = await getipAddress(parseEml.headers['received']);
            let { country, continent } = await getIp.processIp(investigate['senderIp']);
            if (country && continent) {
                investigate['country'] = country;
                investigate['continent'] = continent
            } else {
                investigate['country'] = 'NOT FOUND';
                investigate['continent'] = 'NOT FOUND';
            }
            investigate['spf'] = stringSearcher(authResults, 'spf=pass');
            investigate['dkim'] = stringSearcher(authResults, 'dkim=pass');
            investigate['dmarc'] = stringSearcher(authResults, ['dmarc=pass', 'dmarc=none']);
            return investigate;
        } else {
            console.error('error from parsing header details');
            return {};
        }
    } catch (err) {
        console.log(err);
        return {};
    }
}

const checkUrlsForRid = async (urls) => {
    try {
        let urlContainsRid = false
        const urlsArray = Array.from(urls);

        for(let i=0; i<urlsArray.length;i++){
            let url = urlsArray[i]
            let decodedUrl = decodeURIComponent(url);
            if(decodedUrl.indexOf('rid=')>-1){
                urlContainsRid = decodedUrl
                break;
            }
        }
        return urlContainsRid
        
    } catch (err) {
        console.log(err);
        return false;
    }
}


let controller = async (req, res, next) => {
    try {
        let { messageId, email, internetMessageId, header, body, from, subject, cc, toName = '', bcc = [], mailRecievedTime, mimeContent } = req.body;
        console.log({ messageId, email, internetMessageId, from, subject, cc, toName, mailRecievedTime, });
        console.time(`find-user-${messageId}`)
        email = email.toLowerCase()
        const user = await db.User.findOne({ email, emailType: "Exchange", isActive: true,  deletedAt: null })
        console.timeEnd(`find-user-${messageId}`)
        if (!user) {
            throw new RequestError('user not exists', 404);
        }

        const companyRemainingTime = await db.MetaData.findOne({name:'expiredAt', companyId: user.companyId })
        if(new Date() > new Date(companyRemainingTime.value)) throw new RequestError('License expired', 404)

        const userId = user._id;
        const domainId = user.domainId;
        const companyId = user.companyId;
        const domain = db.Domain.findOne({
            _id: domainId,
            companyId,
            deletedAt: null
        })
        if (!domain) {
            throw new RequestError('domain not exists', 404);
        }


        const isReported = await db.ReportedMail.findOne({ internetMessageId })
        if (isReported) {
            const { isRecoveredEmail, isTrashedEmail, isDeletedEmail, _id } = isReported;
            let safe = isRecoveredEmail === 'false';
            let threat = isDeletedEmail === 'false';
            let spam = isTrashedEmail === 'false';

            if (safe && threat && spam) {
                await db.WhoElse.create({
                    to: email,
                    from: from.emailAddress,
                    subject,
                    whoelseMail: email,
                    internetMessageId,
                    reportedMailId: _id,
                    isTrashed: true,
                    messageId,
                    userId,
                    domainId,
                    companyId,
                });

                await db.ReportedMail.updateOne({ _id }, { $push: { reporters: {email} } });
                if(await canAccessFeature(companyId,"TPIR-RN")){
                    queue.create('send-reported-mail', {
                            to: email,
                            userName: user ? user.name ? user.name :'' :'',
                            from: from.emailAddress,
                            subject,
                            mailRecievedTime,
                            companyId,
                        })
                        .removeOnComplete(true)
                        .priority('high')
                        .save();
                }

                return res.status(200).json({
                    success: true,
                    message: 'Mail Reported Successfully!',
                    reporters: isReported.reporters ? isReported.reporters.length + 2 : 0,
                    moveToJunk: true
                });
            } else {
                return res.status(200).json({ success: true, message: `Mail Already Reported By Someone and has been found ${!safe ? "Safe" : !spam ? "Spam" : "Threat"}!`, moveToJunk: (!spam || !threat) });
            }
        }
        else {
            let headers = ''
            for (let i = 0; i < header.length; i++) {
                headers = headers + header[i].name + ' ' + header[i].value + '\n';
            }
            const parseEml = await parseEmail.parseEml(headers)

            const tsatUrl = await db.MetaData.findOne({
                companyId,
                name: 'tsatUrl'
            })
            const isTsat = await db.MetaData.findOne({
                companyId,
                name: 'isTsat'
            })

            if (mimeContent) {
                const byteCharacters = Buffer.from(mimeContent, 'base64');
                var fileContent = byteCharacters.toString('binary');
                const parsedEML = await parseEmail.parseEml(fileContent);
                var attachments = parsedEML?.attachments ?? [];
                var filename = Date.now() + '.eml';
                // var writeStream = fs.createWriteStream(filePath);
                // writeStream.write(Buffer.from(fileContent, 'binary'))
            }

            let urlsInbody = await getUrls(body)
            let isRidInBody = false
            if(urlsInbody.size > 0 ) isRidInBody = await checkUrlsForRid(urlsInbody)


            if (await canAccessFeature(companyId,"TPIR-TSAT") && isRidInBody != false && isTsat.value === 'true' && tsatUrl) {
                // call to tsat for email reporting

                queue.create('route-email-tsat', { header: JSON.stringify(header), body: decodeUriComponent(`${body}`), tsatUrl})
                .removeOnComplete(true).priority('high').save();    

                if(await canAccessFeature(companyId,"TPIR-RN")){
                    queue.create('send-reported-mail', {
                        to: email,
                        userName: user ? user.name ? user.name :'' :'',
                        from: from.emailAddress,
                        subject,
                        mailRecievedTime,
                        companyId
                    }).removeOnComplete(true).priority('high').save()
                }
                

                if (mimeContent) {    
                    if(await canAccessFeature(companyId,"TPIR-EASOC")){
                    queue.create('send-soc-mail', {
                        to: email,
                        from: from.emailAddress,
                        subject,
                        mailRecievedTime,
                        header: '',
                        body: '',
                        companyId,
                        attachment: true,
                        fileContent
                    }).removeOnComplete(true).priority('high').save()
                }
                }
                else{
                    if(await canAccessFeature(companyId,"TPIR-EASOC")){
                    queue.create('send-soc-mail', {
                        to: email,
                        from: from.emailAddress,
                        subject,
                        mailRecievedTime,
                        header: headers,
                        body,
                        companyId,
                    }).removeOnComplete(true).priority('high').save()
                }
                }

                return res.status(200).json({ success: true, message: 'email reported successfully', moveToJunk:true });
            }

            let otherTo = [];
            let headerDetails = await getParseHeader(headers)
            let to = email.toLowerCase()

            const emailReported = await db.ReportedMail.create({
                from: from.emailAddress, fromName: from.displayName, to, toName:user.name, subject, mailId: messageId, cc,
                bcc, otherTo, messageId, internetMessageId, mailRecievedTime, companyId, fileName: filename,
                mailType: "Exchange", header: '', body:'', userId, headerDetails: await canAccessFeature(companyId,"TPIR-SDD") ? headerDetails : null, domainId, parsedHeader: ''
            })

            Files.uploadReportedMail({_id:emailReported._id , header:headers, body, parsedHeader:parseEml})

            const resolutionTime = Date.now() - new Date(mailRecievedTime).getTime();

            queue.create('upload-siem-logs',{
                to: email,
                from: from.emailAddress,
                subject,
                companyId,
                mailRecievedTime,
                spf: headerDetails?.spf,
                internetMessageId,
                dmarc: headerDetails?.dmarc,
                dkim: headerDetails?.dkim,
                senderIp: headerDetails?.senderIp,
                header,
                reportedUser: email,
                body,
                resolutionTime,
                attachments,
                links: Array.from(urlsInbody),
            }).removeOnComplete(true).priority('high').save();

            if (mimeContent) {
                let isGsuite = false;
                queue.create('email-reported-attachments-gsuite', {
                    attachments,
                    companyId,
                    reportedMailId: emailReported._id,
                }).removeOnComplete(true).priority('high').save();
                // queue.create('send-soc-mail', {
                //     to: email,
                //     from: from.emailAddress,
                //     subject,
                //     mailRecievedTime,
                    // header: '',
                //     body: '',
                //     companyId,
                //     attachment: true,
                //     filename:fileContent,
                //     isGsuite: false
                // }).removeOnComplete(true).priority('high').save()
                if(await canAccessFeature(companyId,"TPIR-EASOC")){
                queue.create('send-soc-mail',{
                    to: email,
                    from: from.emailAddress ? from.emailAddress : from.displayName ? from.displayName : '',
                    subject,
                    mailRecievedTime,
                    header:headers,
                    body,
                    companyId,
                    
                    isGsuite: false
                }).removeOnComplete(true).priority('high').save()
                }
            }

            queue.create('email-language', {
                body,
                reportedMailId: emailReported._id,
            }).removeOnComplete(true).priority('high').save()

            if(await canAccessFeature(companyId,"TPIR-RN")){
            queue
              .create("send-reported-mail", {
                to: email,
                userName: user ? (user.name ? user.name : "") : "",
                from: from.emailAddress,
                subject,
                mailRecievedTime,
                companyId,
              })
              .removeOnComplete(true)
              .priority("high")
              .save();
            }

            if(await canAccessFeature(companyId,"TPIR-SSRE")){
                queue.create('report-email-score-exchange', {
                    headers: headers,
                    reportedMailId: emailReported._id,
                }).removeOnComplete(true).priority('high').save();
            }

            if (!mimeContent) {
                if(await canAccessFeature(companyId,"TPIR-EASOC")){
                queue
                  .create("send-soc-mail", {
                    to: email,
                    from: from.emailAddress,
                    subject,
                    mailRecievedTime,
                    header: headers,
                    body,
                    companyId,
                    isGsuite: false,
                  })
                  .removeOnComplete(true)
                  .priority("high")
                  .save();
                }
            }

            queue.create('check-sender-domain-deceptive', {
                from: from.emailAddress,
                reportedMailId: emailReported._id,
            }).removeOnComplete(true).priority('high').save();

            queue.create('email-reported-url', {
                body,
                companyId,
                reportedMailId: emailReported._id,
            }).removeOnComplete(true).priority('high').save();

            console.log({headerDetails})

            if (headerDetails?.senderIp) {
                console.log("dnsbl se uparrrr")
                queue.create('process-dnsbl', {
                    reportedMailId: emailReported._id,
                    senderIp: headerDetails.senderIp
                }).removeOnComplete(true).save();
            }

            res.status(200).json({ success: true, message: 'email reported successfully',reporters: 1, moveToJunk:true });
        }
    } catch (err) {
        console.log(err);
        next(err);
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post(controller);
export default apiRouter;

