import Joi from 'joi';
import express from 'express';

import { queue } from '../../kue';
import getMailData from '../../service/get-mail-data';
import Spamc from 'spamc';
import config from '../../config';
import { sanitize } from '../../middlewares/sanitizer';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import getUrls from 'get-urls';
import decodeUriComponent from 'decode-uri-component';
import canAccessFeature from '../../service/allowed-features';
import credentials from '../../service/credentials';


// let validator = Joi.object().keys({
//     messageId: Joi.string().required(),
//     accessToken: Joi.string().allow(null).required(),
//     email: Joi.string().email().required(),
//     internetMessageId: Joi.string().required(),
// });

const checkUrlsForRid = async (urls) => {
    try {
        let urlContainsRid = false
        const urlsArray = Array.from(urls);

        for(let i=0; i<urlsArray.length;i++){
            let url = urlsArray[i]
            let decodedUrl = decodeURIComponent(url);
            if(decodedUrl.indexOf('rid=')>-1){
                urlContainsRid = decodedUrl
                break;
            }
        }
        return urlContainsRid
        
    } catch (err) {
        console.log(err);
        return false;
    }
}


const spamc = new Spamc(config.spamassession.host, config.spamassession.port, 10000);
/**
 * @api {post} /api/report/report-email report-email
 * @apiName Report Mail
 * @apiGroup Report
 * @apiDescription Report a mail
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody {String} accessToken       Required accessToken of the mail.
 * @apiBody {String} messageId         Required messageId of the mail.
 * @apiBody {String} email             Required email of the mail.
 *
 * @apiSuccess {Boolean} success    true
 * @apiSuccess {String} message    'email reported successfully'
 * 
  * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *      {
 *          "success":  true,
 *          "message":  'email reported successfully'        
 *      } 
 * 
 * @apiError {Boolean}    success     false
 * @apiError {Boolean}    messages    ["user not exists"]
 * 
 * 
 * @apiErrorExample Error-Response:
 *     HTTP/1.1 404 Not Found
 *       {
 *           "success": false,
 *           "messages": [
 *               "user not exists"
 *           ]
 *       }
 */

const moveEmailToJunkWithDelay = async (messageId, accessToken, email) => {
    setTimeout(async () => {
        try {
            await getMailData.moveEmailToJunk(messageId, accessToken, email);
            console.log(`Email moved to Junk for ${email}`);
        } catch (error) {
            console.error("Error moving email:", error);
        }
    }, 8000); 
};



let controller = async (req, res, next) => {
    try {
        let { accessToken = null, messageId, email, internetMessageId } = req.body;
        const user = await db.User.findOne({ email: email.toLowerCase(), isActive: true, emailType: "Outlook", deletedAt: null })
        if(!user){
            throw new RequestError('user not exists', 404);
        }
        console.log(user)
        const companyRemainingTime = await db.MetaData.findOne({name:'expiredAt', companyId: user.companyId })
        console.log({companyRemainingTime})
        if(new Date() > new Date(companyRemainingTime.value)) throw new RequestError('License expired', 404)

        const userId = user._id;
        const domainId = user.domainId;
        const companyId = user.companyId;
        

        const domain = await db.Domain.findOne({
            _id: domainId,
            companyId,
            deletedAt: null
        })
        console.log(domain)


        if(!domain){
            throw new RequestError('domain not exists', 404);
        }
        let useGraph = false
        if(!accessToken){
            let { token } = await credentials.getCredentials(companyId, domainId)
            accessToken = token.access_token
            useGraph = true;
        }

        var folderId = '';
        if(useGraph){
            folderId = await getMailData.processFolderIdGraph(messageId, accessToken, email)
        }
        else{
            folderId = await getMailData.processFolderId(messageId, accessToken)
        }

        const isReported = await db.ReportedMail.findOne({ internetMessageId })
        if(isReported){
            if(isReported.isRecoveredEmail === 'false'){
                queue.create('process-trash-request-reported-email',{ companyId, name:user.name, email, domainId, internetMessageId, isReported, folderId}).delay(2500).removeOnComplete(true).priority('high').save();   
                moveEmailToJunkWithDelay(messageId, accessToken, email)
                res.status(200).json({ success: true, reporters: isReported.reporters ? isReported.reporters.length + 2 : 0, spamScore: isReported.spamScore, moveToJunk: true });
            }
            else return res.status(200).json({ success: true, message: 'Email already reported by someone and has been found safe!', moveToJunk: false })
        }
        else{
        console.time(`process mail body ${userId}`)
        
        let header;
        let body;
        let attachments;
        let eml;

        if(useGraph){
            eml = await getMailData.processEmlGraph(messageId,accessToken, email);
            header = eml;
            body = await getMailData.processBodyGraph(messageId, accessToken, email);
            attachments = await getMailData.processAttachmentsGraph(messageId, accessToken, email);
            moveEmailToJunkWithDelay(messageId, accessToken, email)
        }
        else{
            header = await getMailData.processHeader(messageId, accessToken);
            body = await getMailData.processBody(messageId, accessToken);
            attachments = await getMailData.processAttachments(messageId, accessToken);
            eml = await getMailData.processEml(messageId,accessToken); 
            moveEmailToJunkWithDelay(messageId, accessToken, email)      
        }

        console.timeEnd(`process mail body ${userId}`)

        const tsatUrl = await db.MetaData.findOne({
            companyId,
            name: 'tsatUrl'
        })
        const isTsat = await db.MetaData.findOne({
            companyId,
            name: 'isTsat'
        })
                
        let isTsatMail = isTsat.value === 'true';
        
        let urlsInbody = await getUrls(body)
        let isRidInBody = false
        if(urlsInbody.size > 0 ) isRidInBody = await checkUrlsForRid(urlsInbody)

        console.log("canAccessFeature:", await canAccessFeature(companyId, "TPIR-TSAT"));
        console.log("isRidInBody:", isRidInBody);
        console.log("isTsatMail:", isTsatMail);
        console.log("tsatUrl.value:", tsatUrl?.value);
        console.log(
        "condition for tsat-tpir------",
        await canAccessFeature(companyId, "TPIR-TSAT") &&
            isRidInBody != false &&
            isTsatMail &&
            tsatUrl.value
        );

        if (await canAccessFeature(companyId,"TPIR-TSAT") && isRidInBody != false && isTsatMail && tsatUrl.value) {
            // call to tsat for email reporting
            queue.create('route-email-tsat', { header, body: decodeUriComponent(`${body}`), tsatUrl })
            .removeOnComplete(true).priority('high').save();
            
            return res.status(200).json({ success: true, message: 'email reported successfully', reporters: 1, moveToJunk: true, spamScore: 5 });
        }
        
        new Promise(async (resolve, reject) => {
            console.time('spamassession')
            if(await canAccessFeature(companyId,"TPIR-SSRE")){
                 spamc.headers(header, function (err, result) {
                    if(err) reject(err);
                    else resolve(result);
                })
            }
            else resolve(0)
        }).then(result => {
            console.log("SpamScore---->",result)
            console.timeEnd('spamassession')
            queue.create('reported-email-outlook', {
                email, 
                header, 
                body, 
                eml,
                userId,
                userName: user ? user.name ? user.name :'' :'', 
                domainId,
                companyId, 
                messageId, 
                internetMessageId,
                attachments, 
                folderId,
                spamScore: result && result.spamScore? result.spamScore: 0,
                isTsatMail: isRidInBody != false
            })
            .removeOnComplete(true).priority('high').save();

        res.status(200).json({ success: true, message: 'email reported successfully', reporters: 1, moveToJunk: true, spamScore: result && result.spamScore? result.spamScore: 0 });
        })   
    }
    } catch(err){
        console.log(err);
        next(err);
    }
}
const apiRouter = express.Router();
apiRouter.route('/').post( 
    // validateBody(validator), 
    controller);
export default apiRouter;

