import Joi from 'joi';
import express from 'express';
import { sanitize } from '../../middlewares/sanitizer';
import { validateBody } from '../../middlewares/validator';
import { queue } from '../../kue';
import { checkCompanyValidityStrategy, jwtStrategy } from '../../middlewares/strategy';
import { updateReportedMailCache } from '../../functions';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import { cleanDictionaryAndSpecialChars } from '../../functions';
import restrictFeature from '../../middlewares/featureBlocker';
import { permission } from '../../middlewares/permission';

const validator = Joi.object({
  reportedMailId: Joi.string().hex().length(24).required(),
});

const createBulkInsertMeta = (domains, reportedMailId) => {
  return domains
    .filter(domain => (domain.domainType === 'Outlook' || domain.domainType === 'Gsuite') && domain.domainTypeForOutlook !== 'EXCHANGE' && domain.isCredential)
    .map(domain => ({
      isProcessing: false,
      isCompleted: false,
      reportedMailId,
      domainId: domain._id,
      companyId: domain.companyId,
    }));
};

const queueEmailJobs = (domains, update) => {
  domains.forEach(domain => {
    if (domain.isCredential) {
      if(domain.domainType == 'Outlook' && domain.domainTypeForOutlook != "EXCHANGE" ){
        queue.create('email-reported-similiar-emails-outlook', {
              reportedMailId: update._id,
              internetMessageId: update.internetMessageId, domainId: domain._id, companyId: update.companyId, 
              subject: update.subject, from: update.from, email:update.to, mailRecievedTime:update.mailRecievedTime
         }).removeOnComplete(true).priority('high').save();          
      }
      else if (domain.domainType == 'Gsuite' ){
        queue.create('email-reported-similiar-emails-gsuite', {
              reportedMailId: update._id,
              internetMessageId:update.internetMessageId, domainId:domain._id, companyId: update.companyId, 
              subject: update.subject, from: update.from, email: update.to
            }).removeOnComplete(true).priority('high').save();
      }
    }
  });
};

const controller = async (req, res, next) => {
  try {
    const { reportedMailId } = req.body;
    const { companyId } = req.user;

    const reportedMail = await db.ReportedMail.findOne({ _id: reportedMailId });
    if (!reportedMail) throw new RequestError('Mail not found!');

    if(!reportedMail || !reportedMail.subject || reportedMail.subject.length === 0 ) throw new RequestError('Subject is empty!')

    const sanitizedSubject = cleanDictionaryAndSpecialChars(reportedMail?.subject);
    

    if (sanitizedSubject.trim() === '') {
      await db.ReportedMail.updateOne(
        { _id: reportedMail._id, companyId },
        {
          syncEndAt: new Date(),
          isSimilarEmailError: false,
          isSimiliarEmailProcessing: false,
          isSimiliarEmailErrorMessage: '',
          isSimiliarEmailCompleted: true,
        }
      );
      await updateReportedMailCache(reportedMailId, {
        isSimiliarEmailProcessing: false,
        isSimiliarEmailCompleted: true,
        syncStartAt: new Date(),
        isSimilarEmailError: false,
        isSimiliarEmailErrorMessage: '',
      });
      return res.status(200).json({ success: true });
    }
    

    if (reportedMail.isSimiliarEmailProcessing) throw new RequestError('Similiar Email search already in progress!');

    const domains = await db.Domain.find({ companyId, isCredential: true, deletedAt: null });
    if (!domains) throw new RequestError('Domain not found!');

    const updateFields = {
      isSimiliarEmailProcessing: true,
      syncStartAt: new Date(),
      isSimilarEmailError: false,
      isSimiliarEmailErrorMessage: '',
    };

    const update = await db.ReportedMail.findOneAndUpdate(
      { _id: reportedMailId },
      { $set: updateFields },
      { new: true }
    );

    await updateReportedMailCache(reportedMailId, {
      isSimiliarEmailProcessing: true,
      syncStartAt: new Date(),
      isSimilarEmailError: false,
      isSimiliarEmailErrorMessage: '',
    });

    const bulkInsertMeta = createBulkInsertMeta(domains, reportedMailId);
    if (bulkInsertMeta.length > 0) {
      try {
        await db.SimiliarEmailMetdata.insertMany(bulkInsertMeta);
        console.log('Metadata created successfully');
      } catch (err) {
        console.error('Error creating metadata for similar emails', err);
      }
    }
    console.log('Queueing email jobs', update.mailRecievedTime);
    queueEmailJobs(domains, update);
    res.status(200).json({ success: true });

  } catch (err) {
    console.error(err);
    if (err instanceof RequestError) {
      await db.ReportedMail.updateOne(
        { _id: req.body.reportedMailId, companyId: req.user.companyId },
        {
          syncEndAt: new Date(),
          isSimilarEmailError: false,
          isSimiliarEmailProcessing: false,
          isSimiliarEmailErrorMessage: '',
          isSimiliarEmailCompleted: true,
        }
      );
    }
    next(err);
  }
};

const apiRouter = express.Router();

apiRouter.route('/')
  .post(htmlsanitize(), jwtStrategy, checkCompanyValidityStrategy, restrictFeature("TPIR-FSEI"), permission("Emails","Write"), validateBody(validator), controller);

export default apiRouter;
