import express from 'express';
import credentials from '../../service/credentials';
import { queue } from '../../kue';
import { getSenderEmail } from '../../service/outlook-webhook-email-services';

const controller = async (req, res, next) => {
  try {
    if (req.query.validationToken) {
      console.log("Received validation request:", req.query.validationToken);
      return res.status(200).send(req.query.validationToken);
    }

    // Deduplicate events within a single webhook payload to avoid enqueueing duplicates
    const seenMessageIds = new Set();
    for (const event of req.body.value) {
      const { subscriptionId, clientState, resource, resourceData } = event;

      let companyId, domainId, webhookRetrievalEmail;
      try {
        const cs = String(clientState || '');
        const firstDash = cs.indexOf('-');
        const secondDash = cs.indexOf('-', firstDash + 1);

        if (firstDash === -1 || secondDash === -1) {
          throw new Error("Invalid clientState format");
        }

        companyId = cs.slice(0, firstDash);
        domainId = cs.slice(firstDash + 1, secondDash);
        webhookRetrievalEmail = cs.slice(secondDash + 1);

        if (!companyId || !domainId || !webhookRetrievalEmail) {
          throw new Error("Invalid clientState values");
        }
      } catch (err) {
        console.error("Error parsing clientState:", err.message, clientState);
        throw new RequestError("Error parsing clientState", err.message, clientState);
      }

      console.log(`🔹 Webhook for Company: ${companyId}, Domain: ${domainId}, Email: ${webhookRetrievalEmail}`);
      console.log(`🔹 Subscription ID: ${subscriptionId}, Resource: ${resource}`);

      const messageId = resourceData?.id;
      if (!messageId) {
        console.warn("No message ID found in webhook data. Skipping.");
        throw new RequestError("No message ID found in webhook data. Skipping");
      }

      if (seenMessageIds.has(messageId)) {
        console.log(`Duplicate messageId ${messageId} within same payload — skipping enqueue.`);
        continue;
      }
      seenMessageIds.add(messageId);

      const credential = await credentials.getCredentials(companyId, domainId);
      if (!credential) {
        console.error("Failed to retrieve access token.");
        throw new RequestError("Failed to retrieve access token.");
      }

      // Encode both for Graph path safety
      const encodedMailbox = encodeURIComponent(webhookRetrievalEmail.trim());
      const encodedMessageId = encodeURIComponent(messageId);

      const from = await getSenderEmail(encodedMailbox, encodedMessageId, credential.token.access_token);
      if (!from) {
        console.warn(`No sender email found for message ${messageId}. Skipping.`);
        throw new RequestError(`No sender email found for message ${messageId}. Skipping.`);
      }

      const user = await db.User.findOne({
        email: String(from).toLowerCase(),
        isActive: true,
        companyId,
        domainId,
        emailType: "Outlook",
        deletedAt: null
      });

      if (!user) {
        console.warn(`User not found for email: ${from}`);
        throw new RequestError(`User not found for email: ${from}`);
      }

      queue.create('webhook-reported-email-outlook-collector', {
        messageId,
        companyId,
        domainId,
        token: credential.token.access_token,
        webhookRetrievalEmail,
        from: user.email,
        user
      })
      .priority('high')
      .removeOnComplete(true)
      .save((err) => {
        if (err) console.error("Failed to enqueue reported-email-collector:", err);
        else console.log(`Enqueued reported-email-collector for message: ${messageId}`);
      });
    }

    return res.status(200).json({ success: true, message: 'Email reported successfully' });
  } catch (err) {
    console.error("Error processing webhook:", err.message);
    return res.status(200).json({ success: true, message: 'Received but failed internally' });
  }
};

const apiRouter = express.Router();
apiRouter.route('/').post(controller);

export default apiRouter;
