import Joi from 'joi';
import express from 'express';

import { sanitize } from '../../middlewares/sanitizer';
import { jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';

let validator = Joi.object().keys({
    page: Joi.number().min(0).required(),
    limit: Joi.number().min(1).required(),
    search: Joi.string().optional().allow(null).empty('')
});

/**
 * @api {get} /api/v1/reported-email/events?page=1&limit=100 events
 * @apiName events
 * @apiGroup reported-email
 * @apiDescription Get Events of the user
 * 
 * @apiVersion 1.0.0
 * 
 * @apiQuery   {String}   page          (Pagination) Required page of the events records
 * @apiQuery   {String}   limit         (Pagination) Required limit of event records to be returned
 * @apiQuery   {String}   [search]      Search for a particulat event
 * 
 * @apiSuccess {String}   success             true
 * @apiSuccess {Array}    results             record of events
 * @apiSuccess {Number}   count               count of events returned
 * 
 * @apiSuccessExample Success-Response:
 *      HTTP/1.1 200 OK 
 * 
 *      {
 *          "success": true,
 *          "results": [
 *              {
 *                  "_id": "63d0f7dcf111d4088dd477cd",
 *                  "from": "mnnitsarvesh@outlook.com",
 *                  "fromName": "Sarvesh Patel",
 *                  "to": "arpit@kdmarc.com",
 *                  "toName": "",
 *                  "subject": "New Content Out For Registration",
 *                  "score": 3.2,
 *                  "mailRecievedTime": "2023-01-24T08:57:41.000Z",
 *                  "mailType": "Outlook",
 *                  "createdAt": "2023-01-25T09:35:24.278Z"
 *              }
 *          ],
 *          "count": 1
 *      }
 * 
 * @apiError {Boolean} success             false
 * @apiError {Array}   messages            "\"page\" is required"
 * 
 * @apiSuccessExample Error-Response:
 *       HTTP/1.1 404 Not Found
 * 
 *      "success":   false,
 *      "messages" : "\"page\" is required"
 * 
 */

let controller = async (req, res, next) => {
    try {
        const companyId = req.user.companyId;
        const { page = 1, limit = 10, search = '' } = req.query;
        const condition = { companyId };
        if(search && search.length){
            const regex = new RegExp(search, "i");
            condition['$or'] = [
                { to: regex }, 
                { from: regex }, 
                { subject: regex }
            ]
        }
        const count = await db.CheckEmailScoreEvent.countDocuments(condition);
        const results = await db.CheckEmailScoreEvent.find(condition)
        .select('from fromName to toName othersTo subject score mailRecievedTime mailType createdAt')
        .sort({'createdAt': -1})
        .skip((page-1)*limit)
        .limit(limit);
        return res.status(200).json({ success: true, results, count });
    } catch(err){
        console.log(err);
        next(err);
    }
}
const apiRouter = express.Router();
apiRouter.route('/').get(htmlsanitize(), validateBody(validator), jwtStrategy, controller);
export default apiRouter;