import Joi from "joi";
import express from "express";

import { sanitize } from "../../middlewares/sanitizer";
import { jwtStrategy } from "../../middlewares/strategy";
import { validateBody } from "../../middlewares/validator";
import { permission } from "../../middlewares/permission";
import { queue } from "../../kue"; 
import { getReportedMailFromCache, cacheReportedMail } from "../../functions";
import s3 from '../../Content/upload_file'
import { htmlsanitize } from "../../middlewares/sanitizeHtml";
import restrictFeature from "../../middlewares/featureBlocker";


let validator = Joi.object().keys({
    reportedMailId: Joi.string().hex().length(24).optional(),
});

/**
 * @api {get} /api/v1/reported-email/get?reportedMailId=63b676c744ed3b8a72386597 get
 * @apiName get
 * @apiGroup reported-email
 * @apiDescription Get all details of reported mail
 * 
 * @apiVersion 1.0.0
 * 
 * @apiQuery   {String}   reportedMailId      Id of the reported mail
 * 
 * @apiSuccess {String}   success             true
 * @apiSuccess {Object}   reportedEmail       Details of the reported mail
 * 
 * @apiSuccessExample Success-Response:
 *      HTTP/1.1 200 OK 
 * 
 *   {
 *     "success":true,
 *     "reportedEmail":
 *      {"_id":"63d0f16c78b9830aa47b3024",
 *      "from":"sarvesh.patel@kratikal.com",
 *      "fromName":"Sarvesh Patel",
 *      "to":"arpit@kdmarc.com",
 *      "toName":"",
 *      "otherTo":[{"address":"arpit@kdmarc.com","name":""}]}] .....,
 *      "countReportedEmails": 1
 *
 *  }
 * 
 * @apiError {Boolean} success             false
 * @apiError {Array}   messages            ["\"reportedMailId\" is not allowed to be empty"]
 * 
 * @apiErrorExample Error-Response:
 *       HTTP/1.1 404 Not Found
 *      {
 *          "success": false,
 *          "messages": [
 *              "\"reportedMailId\" is not allowed to be empty"
 *       |
 * 
 * 
 */

let controller = async (req, res, next) => {
        const { reportedMailId } = req.query;
        const companyId = req.user.companyId;

        try {
            const [company, companyFeatures] = await Promise.all([
                db.Company.findById(companyId).lean(), // Use `lean` to avoid Mongoose overhead
                db.CompanyFeature.find({ companyId }).select('featureCode -_id').lean()
            ]);

            if (!company) throw new RequestError('Company not found!', 404);

            const featureCodes = new Set(companyFeatures.map((feature) => feature.featureCode));

            const reportedMail = await getReportedMailFromCache(reportedMailId);
            if (reportedMail) {
                // Mark the email as read when retrieved from cache
                await db.ReportedMail.findByIdAndUpdate(
                    reportedMailId,
                    {
                        isRead: true,
                        readAt: new Date(),
                        readBy: req.user._id
                    },
                    { new: true }
                );

                // Update the cached version with read status
                reportedMail.isRead = true;
                reportedMail.readAt = new Date();
                reportedMail.readBy = req.user._id;

                // Re-cache the updated version
                queue.create('cache-reported-mail', { reportedEmail: reportedMail })
                    .removeOnComplete(true)
                    .priority('high')
                    .save();

                // Log the read action for audit
                db.AuditLog.create({
                    userId: req.user._id,
                    companyId: req.user.companyId,
                    action: 'MARK_READ',
                    target: 'REPORTED_EMAIL',
                    targetId: reportedMailId,
                    details: {
                        reportedMailId,
                        action: 'read'
                    }
                }).catch(err => console.error('Audit log error:', err));

                const MailAlreadySent = await db.ReportedMail.findById(reportedMailId).select('isBroadcastMail -_id').lean();
                return res.status(200).json({
                    isAlreadyBroadcasted: MailAlreadySent.isBroadcastMail,
                    success: true,
                    reportedEmail: reportedMail,
                    isBroadcast: company.broadcastThreshold > 0,
                });
            }

            const reportedEmail = await s3.fetchReportedMail(reportedMailId);

            // Mark the email as read when retrieved
            await db.ReportedMail.findByIdAndUpdate(
                reportedMailId,
                {
                    isRead: true,
                    readAt: new Date(),
                    readBy: req.user._id
                },
                { new: true }
            );

            // Update the fetched email with read status before processing
            reportedEmail.isRead = true;
            reportedEmail.readAt = new Date();
            reportedEmail.readBy = req.user._id;

            // Log the read action for audit
            db.AuditLog.create({
                userId: req.user._id,
                companyId: req.user.companyId,
                action: 'MARK_READ',
                target: 'REPORTED_EMAIL',
                targetId: reportedMailId,
                details: {
                    reportedMailId,
                    action: 'read'
                }
            }).catch(err => console.error('Audit log error:', err));

            const reportedEmailFeaturesObject = {
                body: "TPIR-VREB",
                parsedHeader: "TPIR-HA",
                headerDetails: "TPIR-SDD",
                senderDomainDeceptive: "TPIR-SDRC",
                dnsbl: "TPIR-DBL",
                spamScore: "TPIR-SSRE",
            };

            for (const [attribute, featureCode] of Object.entries(reportedEmailFeaturesObject)) {
                if (!featureCodes.has(featureCode)) reportedEmail[attribute] = '';
            }

            // Cache the email with updated read status (single cache operation)
            queue.create('cache-reported-mail', { reportedEmail })
                .removeOnComplete(true)
                .priority('high')
                .save();
            
            res.status(200).json({ success: true, reportedEmail, isBroadcast: company.broadcastThreshold > 0 });

        } catch (error) {
            // Handle errors gracefully
            console.error(error);
            res.status(500).json({ success: false, message: error.message });
        }

};
const apiRouter = express.Router();
apiRouter
    .route("/")
    .get(htmlsanitize(),validateBody(validator), jwtStrategy, restrictFeature('TPIR-VRE'), permission('Emails','Read'), controller);
export default apiRouter;


