import Joi from 'joi';
import express from 'express';

import { sanitize } from '../../middlewares/sanitizer';
import { jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { permission } from '../../middlewares/permission';
import { ConfigService } from 'aws-sdk';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';

let validator = Joi.object().keys({
    page: Joi.number().min(0).required(),
    limit: Joi.number().min(1).required(),
    domainId: Joi.string().hex().length(24).optional(),
    search: Joi.string().optional().allow(null).empty(''),
    isTrashedEmail: Joi.boolean().required(),
    isRecoveredEmail: Joi.boolean().required(),
    isDeletedEmail: Joi.boolean().required(),
    last15: Joi.required(),
    last30: Joi.required(),
    custom: Joi.boolean().required(),
    toDate: Joi.date().required(),
    fromDate: Joi.date().required(),
    mailType:Joi.string().optional().allow(null).empty(''),

});

/**
 * @api {post} /api/reported-email/ignore-list ignore-list
 * @apiName ignore-list
 * @apiGroup Report
 * @apiDescription list of all ignored mails of a particular domain or all domains
 * 
 * @apiVersion 1.0.0
 * 
 * @apiSuccess {Boolean}    success                 true
 * @apiSuccess {Object}     reportedEmails          Object of all ignored mails
 * @apiSuccess {Number}     countReportedEmails     count of ignored mails returned
 * 
 * @apiSuccessExample Success-Response:
 *      HTTP/1.1 200 OK 
 * 
 *   {
 *     "success":true,
 *     "reportedEmail":
 *      {"_id":"63d0f16c78b9830aa47b3024",
 *      "from":"sarvesh.patel@kratikal.com",
 *      "fromName":"Sarvesh Patel",
 *      "to":"arpit@kdmarc.com",
 *      "toName":"",
 *      "otherTo":[{"address":"arpit@kdmarc.com","name":""}]} .....
 *  }

 */

let controller = async (req, res, next) => {
    try {
        const { domainId, page = 1, limit = 10, search = '', isRecoveredEmail = false, isTrashedEmail= false, isDeletedEmail = false, mailType, last15, last30, custom, toDate, fromDate } = req.query;
        const companyId = req.user.companyId;

        const company = await db.Company.findById(companyId)
        if(!company) throw new RequestError('Company not found!',404)

        let conditionForDomain = {
            companyId,
            deletedAt: null
        }
        
        if(mailType == 'Exchange'){
            conditionForDomain['domainTypeForOutlook'] = 'EXCHANGE'
        }
        else{
            conditionForDomain['domainTypeForOutlook'] = {$ne: 'EXCHANGE'}
        }

        let domainsOfCurrentCompany = await db.Domain.find(conditionForDomain).select('domainName');

        let uniqueDomains = new Set();
        domainsOfCurrentCompany.map((currentDomain) =>{
            uniqueDomains.add(currentDomain)
        });

        domainsOfCurrentCompany = [...uniqueDomains];
        domainsOfCurrentCompany = domainsOfCurrentCompany.map((value) =>{
            return value._id
        })
    
        let condition = {
            companyId,
            isRecoveredEmail,
            isTrashedEmail,
            isDeletedEmail,
            mailType: { $in: ['Outlook', 'Gsuite' ] },
            domainId : {$in: domainsOfCurrentCompany},
            deletedAt: null,
            isIgnored: true
        }
        if(domainId){
            condition['domainId'] = domainId
        }
        if(mailType){
            condition['mailType'] = mailType
        }
        if(last15 !== 'false'){
            condition['createdAt'] = { $gte: new Date(+last15) };
        }
        if(last30 !== 'false'){
            condition['createdAt'] = { $gte: new Date(+last30) };
        }
        if(custom !== 'false'){
            condition['createdAt'] = { $gte: new Date(fromDate), $lte: new Date(toDate) };
        }

        if(search && search.length){
            const regex = new RegExp(search, "i");
            condition['$or'] = [
                { to: regex }, 
                { from: regex }, 
                { subject: regex }, 
            ]
        }
        console.log(condition)
        const countReportedEmails = await db.ReportedMail.countDocuments(condition)
        const reportedEmails = await db.ReportedMail.find(condition)
        .select('-header -body -messageId -mailId')
        .sort({'createdAt': -1})
        .skip((page-1)*limit)
        .limit(limit);

        for(let i=0; i<reportedEmails.length; i++){
            let emailCount = await db.WhoElse.countDocuments({
                reportedMailId: reportedEmails[i]._doc._id
            });
            reportedEmails[i] = { ...reportedEmails[i]._doc, emailCount };
        }
        res.status(200).json({ success: true, reportedEmails, countReportedEmails, isBroadcast: company.broadcastThreshold > 0 });
    } catch(err){
        console.log(err);
        next(err);
    }
}

const apiRouter = express.Router();
apiRouter.route('/').get(htmlsanitize(), validateBody(validator), jwtStrategy, permission('Emails','Read'), controller);
export default apiRouter;