import Joi from 'joi';
import express from 'express';

import { sanitize } from '../../middlewares/sanitizer';
import { checkCompanyValidityStrategy, jwtStrategy } from '../../middlewares/strategy';
import { validateBody } from '../../middlewares/validator';
import { permission } from '../../middlewares/permission';
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import restrictFeature from '../../middlewares/featureBlocker';

let validator = Joi.object().keys({
    page: Joi.number().min(0).required(),
    limit: Joi.number().min(1).required(),
    search: Joi.string().optional().allow(null).empty(''),
    reportedMailId: Joi.string().hex().length(24).required()
});

/**
 * @api {get} /api/v1/reported-email/who-else?reportedMailId=63b676c744ed3b8a72386597&page=1&limit=100 who-else
 * @apiName events
 * @apiGroup reported-email
 * @apiDescription Get Events of the user
 * 
 * @apiVersion 1.0.0
 * 
 * @apiQuery   {String}   page              (Pagination) Required page of the events records
 * @apiQuery   {String}   limit             (Pagination) Required limit of event records to be returned
 * @apiQuery   {String}   reportedMailId    Required reportedMailId of the mail
 * @apiQuery   {String}   [search]          Search for a particulat who-else record
 * 
 * @apiSuccess {String}   success             true
 * @apiSuccess {Array}    whoElse             who else mails found
 * @apiSuccess {Number}   count               count of who else mails returned
 * 
 * @apiSuccessExample Success-Response:
 *      HTTP/1.1 200 OK 
 *           
 *       {
 *              "success": true,
 *              "whoElse": [{
 *              "_id": "63d0f7f9f111d4088dd477fd",
 *              "to": "pavan@kdmarc.com",
 *              "from": "mnnitsarvesh@outlook.com",
 *              "subject": "New Content Out For Registration",
 *              "whoelseMail": "pavan@kdmarc.com",
 *              "messageId": "AAMkADdlZGQ1MWVkLWQzZTQtNDkzOS1hOGZlLTE2MzI2Y2NiZDMwMABGAAAAAABh1Ro9ZHOaQat_DQuLkrpXBwBQ5-_l0X-MSIhbub19I377AAAAAAEMAABQ5-_l0X-MSIhbub19I377AAOc0ER_AAA=",
 *              "threadId": null,
 *              "isTrashed": false,
 *              "isDeleted": true,
 *              "isErrorWhileMoving": {
 *                  "isError": false,
 *                  "message": ""
 *              },
 *              "userId": "63d0efab032f943fc31e4edb",
 *              "domainId": "63d0eaaa032f943fc31e4ea2",
 *              "companyId": "63d0ea61032f943fc31e4e80",
 *              "reportedMailId": "63d0f7f3f111d4088dd477e0",
 *              "deletedAt": null,
 *              "createdAt": "2023-01-25T09:35:53.541Z",
 *              "updatedAt": "2023-01-25T09:36:47.357Z",
 *              "__v": 0
 *          }],
 *          "count": 1
 *      }
 * 
 * @apiError {Boolean} success         false
 * @apiError {Array}   messages        "\"reportedMailId\" is not allowed to be empty"
 * 
 * @apiSuccessExample Error-Response:
 *       HTTP/1.1 400 Not Found
 * 
 *      "success":   false,
 *      "messages" : "\"reportedMailId\" is not allowed to be empty"
 *   "messages" : "\"page\" is required"
 * 
 */


let controller = async (req, res, next) => {
    try {
        const { reportedMailId, page = 1, limit = 10, search = '' } = req.query;
        const condition = { reportedMailId };
        if(search && search.length){
            const invalidPattern = /[^a-zA-Z0-9@._-]/;  
            if (!invalidPattern.test(search)) {
                const regex = new RegExp(search, "i");
                condition['$or'] = [
                    { to: regex },
                ];
            } else {
                condition['$or'] = [
                    { to: /^$/ },  
                ];
            }
        }
        const reportedMail = await db.ReportedMail.findOne({_id: reportedMailId, companyId: req.user.companyId})
        if(reportedMail.isWhoElseProcessing) return res.status(200).json({ success: true, whoElse:[], count:0 });

        const count = await db.WhoElse.countDocuments(condition);
        const whoElse = await db.WhoElse.find(condition)
        .skip((page-1)*limit)
        .limit(limit);
        res.status(200).json({ success: true, whoElse, count });
    } catch(err){
        console.log(err);
        next(err);
    }
}
const apiRouter = express.Router();
apiRouter.route('/').get(htmlsanitize(), validateBody(validator), jwtStrategy, restrictFeature("TPIR-WE"), permission('Emails','Read'), controller);
export default apiRouter;