import Joi from "joi";
import express from "express";
import { checkCompanyValidityStrategy, checkTaConnectionStrategy, jwtStrategy } from "../../middlewares/strategy";
import { permission } from "../../middlewares/permission";
import { validateBody } from "../../middlewares/validator";
import { htmlsanitize } from "../../middlewares/sanitizeHtml";

const ROLE_NAMES = ["admin", "viewer", "member"];
const ERRORS = {
  ROLE_EXISTS: "Role with the same name already exists",
  ROLE_NOT_FOUND: "Role does not exist",
  MODIFY_DEFAULT: "Cannot modify default roles"
};

const validator = Joi.object().keys({
  policyData: Joi.array().items(Joi.string()).required(),
  roleName: Joi.string().required(),
  roleId: Joi.string().hex().length(24).required(),
  description: Joi.string()
});

const controller = async (req, res, next) => {
  try {
    let { policyData, roleName, roleId, description = '' } = req.body;
    const { companyId } = req.user;
    roleName = roleName.toLowerCase();
    if (ROLE_NAMES.includes(roleName)) {
      throw new RequestError(ERRORS.ROLE_EXISTS);
    }

    const Role = await db.Role.findOne({ _id: roleId, companyId });
    if (!Role) {
      throw new RequestError(ERRORS.ROLE_NOT_FOUND);
    }

    if (Role.roleType === "Default") {
      throw new RequestError(ERRORS.MODIFY_DEFAULT);
    }

    const existingRole = await db.Role.findOne({ roleName, companyId });
    if (existingRole && existingRole._id.toString() !== roleId) {
      throw new RequestError(ERRORS.ROLE_EXISTS);
    }

    const policies = await db.Policy.find({ name: { $in: policyData } });
    const policyIdArr = policies.map(policy => policy._id);

    const updateData = { policyIds: policyIdArr };
    if (description.length > 0) {
      updateData.roleName = roleName;
      updateData.description = description;
    }

    await db.Role.updateOne({ _id: Role._id }, updateData);

    res.status(201).send({ success: true, message: "Role updated successfully" });
  } catch (err) {
    console.error(err);
    next(err);
  }
};

const apiRouter = express.Router();
apiRouter.route("/")
  .post(
    htmlsanitize(),
    validateBody(validator),
    jwtStrategy,
    checkTaConnectionStrategy,
    checkCompanyValidityStrategy,
    permission("IAM", "Write"),
    controller
  );

export default apiRouter;
