import fs from 'fs';
import Joi from 'joi';
import path from 'path';
import express from 'express';
import FormData from 'form-data';

import { sanitize } from '../../middlewares/sanitizer';
import { permission } from '../../middlewares/permission'; 
import { checkCompanyValidityStrategy, jwtStrategy } from '../../middlewares/strategy';
import virustotal from '../../service/get-virus-totalkey';
import { validateBody } from '../../middlewares/validator';
import scanAttachments from '../../service/scan-attachments';
import s3 from '../../Content/upload_file'
import { htmlsanitize } from '../../middlewares/sanitizeHtml';
import restrictFeature from '../../middlewares/featureBlocker';

let validator = Joi.object().keys({
    attachmentIds: Joi.array().items(
        Joi.string().hex().length(24).required()
    ).required(),
    reportedMailId: Joi.string().hex().length(24).required()
});

/**
 * @api {post} /api/v1/scan/scan-attachment scan-attachment
 * @apiName scan-attachment
 * @apiGroup scan
 * @apiDescription Scan a attachment
 * 
 * @apiVersion 1.0.0
 * 
 * @apiBody   {String}   attachmentId        AttchmentId of the reported mail
 * @apiBody   {String}   reportedMailId      Id of the reported mail
 * 
 * @apiSuccess {Boolean} Success             true
 * @apiSuccess {String}  Message             Scan now in processing, Result Will be Updated Soon!
 * 
 * @apiSuccessExample Success-Response:
 *      HTTP/1.1 200 OK 
 * {
 *           "Success": true,
 *           "Message": "Scan now in processing, Result Will be Updated Soon!"
 * }
 * 
 * @apiError {Boolean} Success             false
 * @apiError {Array}   messages            "\"reportedMailId\" is not allowed to be empty"
 * 
 * @apiSuccessExample Error-Response:
 *       HTTP/1.1 400 Not Found
 * 
 *      "success":   false,
 *      "messages" : "\"reportedMailId\" is not allowed to be empty"
 * 
 */

let controller = async (req, res, next) => {
	try {
		const { reportedMailId, attachmentIds } = req.body;
		const companyId = req.user.companyId;
		let apikey = await virustotal.getApiKey(companyId);

		for (const attachmentId of attachmentIds) {
			const attachment = await db.Attachment.findOne({
				_id: attachmentId, reportedMailId
			});
			if (!attachment) {
				throw new RequestError('Invalid attachmentId');
			}

			let file = await s3.fetchReportedMailAttachments(reportedMailId, attachment.fileName, apikey)
			if(!file) throw new RequestError('File does not exist');


			let formData = new FormData();
			formData.append('apikey', apikey);
			formData.append('file', file, {filename: attachment.fileName});

			scanAttachments.processScanAttachment(formData, attachmentId, companyId);
			attachment.status = 'IN PROGRESS';
			await attachment.save();
		}

		return res.status(200).send({
			success: true,
			message: "Scans are now in processing, results will be updated soon!"
		});
	} catch (error) {
		console.log("Main Catch", error);
		next(error);
	}
}

const apiRouter = express.Router();
apiRouter.route('/').post(htmlsanitize(), validateBody(validator), jwtStrategy, restrictFeature("TPIR-AALP"), checkCompanyValidityStrategy ,permission('Emails','Write'), controller);
export default apiRouter;